const express = require("express");
const fs = require("fs");
const multer = require("multer");
const mysql = require("mysql2");
const xlsx = require("xlsx");
const path = require("path");
const router = express.Router();
const app = express();
app.use(express.json());

const { promisePool } = require("./db");
const { getPoolByUserId } = require('./databaseSelector');

const { route } = require("./SvgDatabase");

const formatToMySQLDate = (dateString) => {
  const [day, month, year] = dateString.split("/"); // DD/MM/YYYY formatını böl
  return `${year}-${month}-${day}`; // YYYY-MM-DD formatında geri döndür
};
const formatToMySQLDateShop = (dateString) => {
  const [month, day, year] = dateString.split("/"); // DD/MM/YYYY formatını böl
  return `${year}-${month}-${day}`; // YYYY-MM-DD formatında geri döndür
};
router.post("/SaveFormData", async (req, res) => {
  const {
    UserID,
    username,
    selectedShop,
    startDate,
    startTime,
    endTime,
    SelectedEfforts,
    description,
    storeLinks,
    sellerLinks,
    Image1,
    Image2,
    Image3,
    TimeSummary,
    Piace
  } = req.body;

  const pool = await getPoolByUserId(UserID);

  try {
    const formattedDate = formatToMySQLDate(startDate);

    const [conflictRows] = await pool.query(
      `SELECT FormID FROM UserEfforts 
       WHERE UserID = ? 
       AND EffortDate = ? 
       AND (
         (StartHour <= ? AND EndHour > ?) OR 
         (StartHour < ? AND EndHour >= ?) OR
         (? <= StartHour AND ? > StartHour)
       ) LIMIT 1`,
      [UserID, formattedDate, startTime, startTime, endTime, endTime, startTime, endTime]
    );

    if (conflictRows.length > 0) {
      return res.status(400).json({
        success: false,
        message: "Bu saat aralığında zaten bir efor kaydedilmiş. Lütfen farklı bir zaman seçin."
      });
    }

    // FormID belirleme
    const [existingDateRows] = await pool.query(
      `SELECT FormID FROM UserEfforts WHERE EffortDate = ? AND UserID = ? LIMIT 1`,
      [formattedDate, UserID]
    );

    let formID;

    if (existingDateRows.length > 0) {
      formID = existingDateRows[0].FormID;
    } else {
      const [allRows] = await pool.query(
        `SELECT MAX(FormID) as MaxFormID FROM UserEfforts`
      );
      const maxFormID = allRows[0].MaxFormID;
      formID = maxFormID === null ? 1 : maxFormID + 1;
    }

    const insertQuery = `
        INSERT INTO UserEfforts 
        (UserID, UserName, EffortDate, StartHour, EndHour, TimeSummary, SelectedEfforts, Explanation, ShopLinks, PurchLinks, Image1, Image2, Image3, FormID, Shop, Piace) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `;

    await pool.query(insertQuery, [
      UserID,
      username,
      formattedDate,
      startTime,
      endTime,
      TimeSummary,
      SelectedEfforts,
      description,
      storeLinks,
      sellerLinks,
      Image1 || null,
      Image2 || null,
      Image3 || null,
      formID,
      selectedShop,
      Piace
    ]);

    res.json({
      success: true,
      message: "Data saved successfully.",
      FormID: formID,
    });
  } catch (error) {
    console.error("Error saving form data:", error);
    res.status(500).json({ success: false, message: "Error saving data." });
  }
});

router.post("/UpdateFormData", async (req, res) => {
  const {
    indexID, // Güncellenecek satırın ID'si
    UserID,
    username,
    selectedShop,
    startDate,
    startTime,
    endTime,
    SelectedEfforts,
    description,
    storeLinks,
    sellerLinks,
    TimeSummary,
    Piace,
  } = req.body;

  const pool = await getPoolByUserId(UserID);

  try {
    const formattedDate = formatToMySQLDate(startDate); // Tarihi uygun formata çevir

    console.log(indexID);
    const [conflictRows] = await pool.query(
      `SELECT FormID FROM UserEfforts 
      
       WHERE UserID = ? 
       AND Id != ?
       AND EffortDate = ? 
       AND (
         (StartHour <= ? AND EndHour > ?) OR 
         (StartHour < ? AND EndHour >= ?) OR
         (? <= StartHour AND ? > StartHour)
       ) LIMIT 1`,
      [UserID,indexID, formattedDate, startTime, startTime, endTime, endTime, startTime, endTime]
    );

    if (conflictRows.length > 0) {
      return res.status(400).json({
        success: false,
        message: "Bu saat aralığında zaten bir efor kaydedilmiş. Lütfen farklı bir zaman seçin."
      });
    }

    const updateFields = [];
    const updateValues = [];

    if (UserID) {
      updateFields.push("UserID = ?");
      updateValues.push(UserID);
    }
    if (username) {
      updateFields.push("UserName = ?");
      updateValues.push(username);
    }
    if (formattedDate) {
      updateFields.push("EffortDate = ?");
      updateValues.push(formattedDate);
    }
    if (startTime) {
      updateFields.push("StartHour = ?");
      updateValues.push(startTime);
    }
    if (endTime) {
      updateFields.push("EndHour = ?");
      updateValues.push(endTime);
    }
    if (TimeSummary) {
      updateFields.push("TimeSummary = ?");
      updateValues.push(TimeSummary);
    }
    if (description) {
      updateFields.push("Explanation = ?");
      updateValues.push(description);
    }
    if (storeLinks) {
      updateFields.push("ShopLinks = ?");
      updateValues.push(storeLinks);
    }
    if (sellerLinks) {
      updateFields.push("PurchLinks = ?");
      updateValues.push(sellerLinks);
    }
    if (selectedShop) {
      updateFields.push("Shop = ?");
      updateValues.push(selectedShop);
    }
    if (SelectedEfforts) {
      updateFields.push("SelectedEfforts = ?");
      updateValues.push(SelectedEfforts);
    }
    if (Piace !== undefined && Piace !== null && Piace !== "") {
      updateFields.push("Piace = ?");
      updateValues.push(Piace);
    }

    if (updateFields.length > 0) {
      const updateQuery = `
        UPDATE UserEfforts
        SET ${updateFields.join(", ")}
        WHERE Id = ?
      `;
      updateValues.push(indexID);

      const [updateResult] = await pool.query(updateQuery, updateValues);

      if (updateResult.affectedRows > 0) {
        res.json({ success: true, message: "Data updated successfully." });
      } else {
        res
          .status(404)
          .json({
            success: false,
            message: "Data not found for the provided ID.",
          });
      }
    } else {
      res
        .status(400)
        .json({ success: false, message: "No valid fields provided to update." });
    }
  } catch (error) {
    console.error("Error updating form data:", error);
    res.status(500).json({ success: false, message: "Error updating data." });
  }
});

router.post("/GetDailyEffort", async (req, res) => {
  try {
    const { UserID, date, UserIsAdmin } = req.body;

   const pool = await getPoolByUserId(UserID);

    let sqlQuery = "";
    let queryParams = [];

    if (UserIsAdmin === 1) {
      sqlQuery = `SELECT * FROM UserEfforts WHERE EffortDate = ?`;
      queryParams = [date];
    } else {
      sqlQuery = `SELECT * FROM UserEfforts WHERE EffortDate = ? AND UserID = ?`;
      queryParams = [date, UserID];
    }

    const [existingDateRows] = await pool.query(sqlQuery, queryParams);

    res.status(200).json(existingDateRows);
  } catch (error) {
    console.error("GetDailyEffort Hatası:", error);
    res
      .status(500)
      .json({ error: "Günlük efor verileri alınırken bir hata oluştu." });
  }
});

router.post("/SaveShopFees", async (req, res) => {
  const {
    UserID,
    username,
    selectedShop,
    startDate,
    endDate,
    salesSummary,
    totalSalesAmount,
    totalSales,
    totalRefundsAmounts,
    totalRefunds,
    listingFees,
    marketingFees,
    ShippingFees,
    CSVSalesSummary,
    filterStartDateCSV,
    filterEndDateCSV,
    productTotalSum,
    productCustomCost,
    returnAndExchange,
    piaceInput,
    rows,
    ShippingFeesCustomHub,
  } = req.body;

   const pool = await getPoolByUserId(UserID);

  // Transaction başlat
  const connection = await pool.getConnection();
  await connection.beginTransaction();

  try {
    // Format startDate and endDate to MySQL DATE format if necessary
    const formattedStartDate = formatToMySQLDateShop(startDate);
    const formattedEndDate = formatToMySQLDateShop(endDate);
    const formattedFilterStartDateCSV = formatToMySQLDateShop(filterStartDateCSV);
    const formattedFilterEndDateCSV = formatToMySQLDateShop(filterEndDateCSV);

    const [conflictRows] = await connection.query(
      `SELECT FormID FROM ShopFees 
       WHERE SelectedShop = ?
       AND (
           (StartDate <= ? AND EndDate >= ?) 
           OR (StartDate <= ? AND EndDate >= ?) 
           OR (? <= StartDate AND ? >= EndDate) 
       )
       LIMIT 1;`,
      [selectedShop, formattedStartDate, formattedStartDate, formattedEndDate, formattedEndDate, formattedStartDate, formattedEndDate]
    );

    if (conflictRows.length > 0) {
      return res.status(400).json({
        success: false,
        message: "Bu tarih aralığında zaten bir kayıt mevcut. Lütfen farklı bir tarih seçin."
      });
    }
    
    const [existingRows] = await connection.query(
      `SELECT FormID FROM ShopFees WHERE StartDate = ? AND EndDate = ? AND UserID = ? LIMIT 1`,
      [formattedStartDate, formattedEndDate, UserID]
    );

    let formID;

    if (existingRows.length > 0) {
      // Use existing FormID
      formID = existingRows[0].FormID;
    } else {
      // Generate new FormID
      const [allRows] = await connection.query(
        `SELECT MAX(FormID) AS MaxFormID FROM ShopFees`
      );
      const maxFormID = allRows[0].MaxFormID;
      formID = maxFormID ? maxFormID + 1 : 1; // Increment or start at 1
    }

    // Insert or update the record
    const insertQuery = `
      INSERT INTO ShopFees (
        FormID, 
        UserID, 
        Username, 
        SelectedShop, 
        StartDate, 
        EndDate, 
        salesSummary,
        totalSalesAmount, 
        totalSales, 
        totalRefundsAmounts, 
        totalRefunds, 
        listingFees,
        marketingFees,
        ShippingFees,
        CSVSalesSummary,
        productTotalSum,
        productCustomCost,
        returnAndExchange,
        piaceInput,
        filterStartDateCSV,
        filterEndDateCSV,
        ShippingFeesCSV
      ) 
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?) 
      ON DUPLICATE KEY UPDATE 
        salesSummary = VALUES(salesSummary),
        totalSalesAmount = VALUES(totalSalesAmount),
        totalSales = VALUES(totalSales),
        totalRefundsAmounts = VALUES(totalRefundsAmounts),
        totalRefunds = VALUES(totalRefunds),
        listingFees = VALUES(listingFees),
        marketingFees = VALUES(marketingFees),
        ShippingFees = VALUES(ShippingFees),
        CSVSalesSummary = VALUES(CSVSalesSummary),
        productTotalSum = VALUES(productTotalSum),
        productCustomCost = VALUES(productCustomCost),
        returnAndExchange = VALUES(returnAndExchange),
        piaceInput = VALUES(piaceInput),
        SelectedShop = VALUES(SelectedShop),
        Username = VALUES(Username),
        ShippingFeesCSV = VALUES(ShippingFeesCSV)
    `;

    const [result] = await connection.query(insertQuery, [
      formID,
      UserID,
      username,
      selectedShop,
      formattedStartDate,
      formattedEndDate,
      salesSummary || "",
      totalSalesAmount || 0,
      totalSales || 0,
      totalRefundsAmounts || 0,
      totalRefunds || 0,
      listingFees || 0,
      marketingFees || 0,
      ShippingFees || 0,
      CSVSalesSummary || "",
      productTotalSum || 0,
      productCustomCost || 0,
      returnAndExchange || 0,
      piaceInput || 0,
      formattedFilterStartDateCSV,
      formattedFilterEndDateCSV,
      ShippingFeesCustomHub,
    ]);

    const insertedID = result.insertId;

    if (rows && rows.length > 0) {
      const insertEtsyTableQuery = `
        INSERT INTO ShopFeesEtsyTable (PaymentDate, PaymentFee, ParentFormID)
        VALUES (?, ?, ?)
      `;
    
      for (const row of rows) {
        // Eğer tarih boş veya geçersizse, döngünün o adımını atla
        if (!row.date || isNaN(new Date(row.date))) {
          continue;
        }
    
        const dateObj = new Date(row.date);
        dateObj.setDate(dateObj.getDate());
        const dateOnly = dateObj.toISOString().split("T")[0];
        const amount = parseFloat(row.amount.replace(".", "").replace(",", "."));
    
        await connection.query(insertEtsyTableQuery, [dateOnly, amount, insertedID]);
      }
    }

    await connection.commit();

    res.json({
      success: true,
      message: "Shop fees data saved successfully.",
      FormID: formID,
    });
  } catch (error) {
    await connection.rollback();
    console.error("Error saving shop fees data:", error);
    res.status(500).json({ success: false, message: "Error saving data." });
  } finally {
    connection.release();
  }
});

router.post("/UpdateShopFees", async (req, res) => {
  const {
    UserID,
    username,
    selectedShop,
    startDate,
    endDate,
    salesSummary,
    totalSalesAmount,
    totalSales,
    totalRefundsAmounts,
    totalRefunds,
    listingFees,
    marketingFees,
    ShippingFees,
    CSVSalesSummary,
    filterStartDateCSV,
    filterEndDateCSV,
    productTotalSum,
    productCustomCost,
    returnAndExchange,
    piaceInput,
    formID,
    rowsUpdate,
    ShippingFeesCustomHub
  } = req.body;
  try {
    const formattedStartDate = formatToMySQLDateShop(startDate);
    const formattedEndDate = formatToMySQLDateShop(endDate);
    const formattedFilterStartDateCSV = formatToMySQLDateShop(filterStartDateCSV);
    const formattedFilterEndDateCSV = formatToMySQLDateShop(filterEndDateCSV);
    
    const pool = await getPoolByUserId(UserID);

   const updateQuery = `
    UPDATE ShopFees 
    SET 
    UserID = ?,
    Username = ?,
    SelectedShop = ?,
    StartDate = ?,
    EndDate = ?,
    salesSummary = ?,
    totalSalesAmount = ?,
    totalSales = ?,
    totalRefundsAmounts = ?,
    totalRefunds = ?,
    listingFees = ?,
    marketingFees = ?,
    ShippingFees = ?,
    CSVSalesSummary = ?,
    productTotalSum = ?,
    productCustomCost = ?,
    returnAndExchange = ?,
    piaceInput = ?,
    filterStartDateCSV = ?,
    filterEndDateCSV = ?,
    ShippingFeesCSV = ?
  WHERE ID = ?
`;

await pool.query(updateQuery, [
  UserID,
  username,
  selectedShop,
  formattedStartDate,
  formattedEndDate,
  salesSummary || '',
  totalSalesAmount || 0,
  totalSales || 0,
  totalRefundsAmounts || 0,
  totalRefunds || 0,
  listingFees || 0,
  marketingFees || 0,
  ShippingFees || 0,
  CSVSalesSummary || '',
  productTotalSum || 0,
  productCustomCost || 0,
  returnAndExchange || 0,
  piaceInput || 0,
  formattedFilterStartDateCSV,
  formattedFilterEndDateCSV,
  ShippingFeesCustomHub,
  formID 
]);

    const deleteQuery =`
      delete from ShopFeesEtsyTable
      WHERE ParentFormID = ?
    `
    await pool.query(deleteQuery,[formID]);

    const insertEtsyTableQuery = `
    INSERT INTO ShopFeesEtsyTable (PaymentDate, PaymentFee, ParentFormID)
    VALUES (?, ?, ?)
    `;
    
    if (Array.isArray(rowsUpdate) && rowsUpdate.length > 0) {
      for (const row of rowsUpdate) {
        console.log(row + 'RowsUpdate');
        if (!row.date || !row.amount) {
          console.warn("Skipping row with missing date or amount:", row);
          continue; 
        }
    
        const dateObj = new Date(row.date);
        console.log(dateObj);
        dateObj.setDate(dateObj.getDate());
        const dateOnly = dateObj.toISOString().split('T')[0];
        
        let amount = parseFloat(row.amount.replace('.', '').replace(',', '.'));
        if (isNaN(amount) || amount === undefined || amount === null) {
          amount = 0; 
        }
    
        await pool.query(insertEtsyTableQuery, [dateOnly, amount, formID]);
      }
    }
    res.json({
      success: true,
      message: "Shop fees data saved successfully.",
      FormID: formID,
    });

  } catch (error) {
    console.error("Error saving shop fees data:", error);
    res.status(500).json({ success: false, message: "Error saving data." });
  }
});

router.post("/DeleteShopFees", async (req, res) => {
  try {
    const { formID, UserID } = req.body;

    const pool = await getPoolByUserId(UserID);

    if (!formID) {
      return res.status(400).json({ success: false, message: "formID gereklidir." });
    }

    // Önce ShopFeesEtsyTable içindeki ilişkili verileri sil
    const deleteEtsyQuery = `
      DELETE FROM ShopFeesEtsyTable 
      WHERE ParentFormID = ?
    `;
    await pool.query(deleteEtsyQuery, [formID]);

    // Ardından ShopFees içindeki ana kaydı sil
    const deleteShopFeesQuery = `
      DELETE FROM ShopFees 
      WHERE ID = ?
    `;
    const [result] = await pool.query(deleteShopFeesQuery, [formID]);
    console.log(formID);
    if (result.affectedRows === 0) {
      return res.status(404).json({ success: false, message: "Belirtilen formID ile eşleşen kayıt bulunamadı." });
    }

    res.json({ success: true, message: "Shop verisi başarıyla silindi.", FormID: formID });
  } catch (error) {
    console.error("Shop verisi silinirken hata oluştu:", error);
    res.status(500).json({ success: false, message: "Veri silinirken bir hata oluştu." });
  }
});

router.post("/GetShopFeesData", async (req, res) => {
  try {
    const { UserID, UserIsAdmin, date } = req.body;

    const pool = await getPoolByUserId(UserID);
    console.log(UserIsAdmin + 'Date');
    let query = `
      SELECT 
        sf.*,
        sft.ID AS EtsyID,
        sft.PaymentDate,
        sft.PaymentFee
      FROM ShopFees sf
      LEFT JOIN ShopFeesEtsyTable sft ON sf.ID = sft.ParentFormID
      WHERE (? BETWEEN sf.StartDate AND sf.EndDate OR ? = sf.StartDate OR ? = sf.EndDate)
    `;

    let queryParams = [date, date, date]; 

    if (UserIsAdmin !== 1) {
        query += " AND SelectedShop in (select ShopName from ShopsAuth where UserID = ?)"; 
        queryParams.push(UserID); 
    }

    query += " ORDER BY sf.ID, sft.PaymentDate ASC";

    const [rows] = await pool.query(query, queryParams);
    console.log(rows);
    res.status(200).json(rows);
  } catch (error) {
      console.error("GetShopFeesData Hatası:", error);
      res.status(500).json({ error: "Veriler alınırken bir hata oluştu." });
  }
});

router.post("/GetMonthlyUserEfforts", async (req, res) => {

  try {
    const { UserID, month, year } = req.body;

    const pool = await getPoolByUserId(UserID);
    // SQL sorgusunu çalıştır
    const [totalEffort] = await pool.query(
      `SELECT 
        UserID,
        CONCAT(
          FLOOR(
            SUM(
              SUBSTRING_INDEX(TimeSummary, ':', 1) * 60 + 
              SUBSTRING_INDEX(TimeSummary, ':', -1)
            ) / 60
          ),
          ':',
          LPAD(
            MOD(
              SUM(
                SUBSTRING_INDEX(TimeSummary, ':', 1) * 60 + 
                SUBSTRING_INDEX(TimeSummary, ':', -1)
              ), 60
            ),
            2,
            '0'
          )
        ) as TotalEffort
      FROM 
        UserEfforts
      WHERE 
        UserID = ?
        AND MONTH(EffortDate) = ? 
        AND YEAR(EffortDate) = ?
      GROUP BY 
        UserID`,
      [UserID, month, year]
    );
    // Sonuçları frontend'e JSON formatında gönder
    res.status(200).json(totalEffort);
  } catch (error) {
    console.error("GetUserEfforts Hatası:", error);
    res.status(500).json({ error: "Toplam efor hesaplanırken bir hata oluştu." });
  }
});

router.post("/getUserChecks", async (req, res) => {
  const { UserID, selectedDate } = req.body;

  const pool = await getPoolByUserId(UserID);

  try{
    const [UserChecks] = await pool.query(
      ` SELECT CheckBoxType, CheckDate 
    FROM UserCheckProgress 
    WHERE UserID = ? AND DATE(CheckDate) = DATE(?)`,
      [UserID, selectedDate]
    );

    // Sonuçları frontend'e JSON formatında gönder
    res.status(200).json(UserChecks);
  }
  catch (error) {
    console.error("UserChecks Hatası:", error);
    res.status(500).json({ error: "Veriler alınırken bir hata oluştu." });
  }

});

router.post("/insertCheck", async (req, res) => {
  const { UserID, UserName, CheckBoxType, CheckDate } = req.body;

  const pool = await getPoolByUserId(UserID);
  // Gelen tarihi MySQL formatına dönüştür
  const formattedDate = new Date(CheckDate).toISOString().slice(0, 19).replace("T", " ");
  
  try {
    const [insertCheck] = await pool.query(
      `INSERT INTO UserCheckProgress (UserID, UserName, CheckBoxType, CheckDate)
       VALUES (?, ?, ?, ?)
       ON DUPLICATE KEY UPDATE CheckDate = VALUES(CheckDate);`,
      [UserID, UserName, CheckBoxType, formattedDate] // Formatlanmış tarihi kullan
    );

    // Sonuçları frontend'e JSON formatında gönder
    res.status(200).json(insertCheck);
  } catch (error) {
    console.error("UserChecks Hatası:", error);
    res.status(500).json({ error: "Veriler alınırken bir hata oluştu." });
  }
});


module.exports = router;
