const express = require("express");
const fs = require("fs");
const multer = require("multer");
const mysql = require("mysql2");
const xlsx = require("xlsx");
const path = require("path");
const router = express.Router();
const app = express();
app.use(express.json());

const { promisePool } = require("./db");
const { getPoolByUserId } = require('./databaseSelector');

const formatDate = (date) => {
  if (!date) return '';
  const d = new Date(date);
  const day = String(d.getDate()).padStart(2, '0');
  const month = String(d.getMonth() + 1).padStart(2, '0'); // Months are 0-based
  const year = d.getFullYear();
  return `${year}-${month}-${day}`;
};
const formatToMySQLDateShop = (dateString) => {
  const [month, day, year] = dateString.split("/"); // DD/MM/YYYY formatını böl
  return `${year}-${month}-${day}`; // YYYY-MM-DD formatında geri döndür
};

router.post("/GetUserData", async (req, res) => {
  try {
    const { UserID } = req.body;

    const [authResult] = await promisePool.query(
      'SELECT UbertySecondDatabase FROM ButtonAuth WHERE UserID = ?',
      [UserID]
    );

    if (!authResult.length) {
      return res.status(404).json({ error: "Kullanıcı bulunamadı" });
    }

    const canViewUberty = authResult[0].UbertySecondDatabase === 1;

    let query = `
      SELECT us.UserID, us.UserName, us.Email
      FROM Users us
      INNER JOIN ButtonAuth ba ON us.UserID = ba.UserID
      WHERE us.Status = 1
    `;

    const params = [];

    if (canViewUberty) {
      query += ` AND ba.UbertySecondDatabase = 1`;
    }

    const [Users] = await promisePool.query(query, params);

    res.status(200).json(Users);
  } catch (error) {
    console.error("GetUserData Hatası:", error);
    res.status(500).json({ error: "Veriler alınırken bir hata oluştu." });
  }
});

router.post("/saveUserShops", async (req, res) => {

  const { UserID,selectedUserID,payload  } = req.body;
  const pool = await getPoolByUserId(UserID);

  if (!UserID) {
    return res.status(400).json({ error: "UserID is required" });
  }
  if (!Array.isArray(payload)) {
    return res.status(400).json({ error: "assignments must be an array" });
  }
  try {
    const connection = await pool.getConnection();
    try {

    if (!selectedUserID) {
      return res.status(400).json({ error: "Payload boş veya geçersiz" });
    }

    const deleteSql = `DELETE FROM ShopsAuth WHERE UserID = ?`;
    await connection.query(deleteSql, [selectedUserID]);
    console.log('UserID:', selectedUserID);
  } catch (error) {
    console.error("Error saving user shops:", error);
  }

   
    console.log('Deleted existing user shops for UserID:', UserID);
    if (payload.length > 0) {
      const values = payload.map((item) => [
        item.ShopName,
        item.ShopID,
        item.UserID,
        item.UserName,
        item.Status,
      ]);

      const insertSql = `
        INSERT INTO ShopsAuth (ShopName, ShopID, UserID, UserName, Status)
        VALUES ?
      `;

      await connection.query(insertSql, [values]);
    }

    connection.release();
    return res.status(200).json({ success: true });
  } catch (error) {
    console.error("Error saving user shops:", error);
    return res.status(500).json({ error: "Error saving user shops" });
  }
});

router.post("/saveUserHierarchy", async (req, res) => {

  const {UserID, userShopAssignments} = req.body;
  
  const pool = await getPoolByUserId(UserID);

  if (!Array.isArray(userShopAssignments)) {
    return res.status(400).json({ error: "Body must be an array" });
  }

  if (userShopAssignments.length === 0) {
    return res.status(400).json({ error: "No shop assignments provided." });
  }

  try {
    const connection = await pool.getConnection();

    const userId = userShopAssignments[0].CaptainUserID;
    const deleteSql = `DELETE FROM UserAuth WHERE CaptainUserID = ?`;
    await connection.query(deleteSql, [userId]);
    console.log('userId:', userShopAssignments[0].CaptainUserID);

    const values = userShopAssignments.map((item) => [
      item.StaffUserName,
      item.StaffUserID,
      item.CaptainUserID,
      item.CaptainUserName,
      item.Status,
    ]);

    const insertSql = `
        INSERT INTO UserAuth (StaffUserName, StaffUserID, CaptainUserID, CaptainUserName, Status)
        VALUES ?
      `;

    await connection.query(insertSql, [values]);

    connection.release();
    return res.status(200).json({ success: true });
  } catch (error) {
    console.error("Error saving user shops:", error);
    return res.status(500).json({ error: "Error saving user shops" });
  }
});

router.post("/getUserShops", async (req, res) => {
  try {
    const { UserID, UserIsAdmin } = req.body;

    const pool = await getPoolByUserId(UserID);

    const connection = await pool.getConnection();

    let getShopAuth;
    let queryParams = [];
    console.log('UserIsAdmin:', UserIsAdmin);
    if (UserIsAdmin === 1) {
      getShopAuth = `SELECT * FROM ShopsAuth`;
    } else {

      getShopAuth = `SELECT * FROM ShopsAuth WHERE UserID = ?`;
      queryParams.push(UserID);
    }

    const [rows] = await connection.query(getShopAuth, queryParams);
    connection.release();
    console.log('rows:', rows);
    return res.status(200).json({ data: rows });
  } catch (error) {
    console.error("Error retrieving user shops:", error);
    return res.status(500).json({ error: "Error retrieving user shops" });
  }
});

router.post("/getUserUsers", async (req, res) => {
    try {
      const { UserID } = req.body; 
      const pool = await getPoolByUserId(UserID);
      const connection = await pool.getConnection();
  
      const getShopAuth = `
        SELECT * FROM UserAuth 
        WHERE CaptainUserID = ?
      `;
      const [rows] = await connection.query(getShopAuth, [UserID]);
  
      connection.release();
  
      // rows => ShopsAuth tablosundan gelen satırlar
      return res.status(200).json({  data: rows });
    } catch (error) {
      console.error("Error saving user shops:", error);
      return res.status(500).json({ error: "Error saving user shops" });
    }
  });

router.post('/effortsFees', async (req, res) => {

    const {efforts,UserID} = req.body;

    const pool = await getPoolByUserId(UserID);

    try {
      const connection = await pool.getConnection();
      
      try {
        await connection.beginTransaction();
        
        const deleteSql = `DELETE FROM UserEffortsFees`;
        await connection.query(deleteSql);

        for (const effort of efforts) {
          if(effort.effortType1Fee === null || effort.effortType1Fee === undefined || effort.effortType1Fee === '') {
            effort.effortType1Fee = 0;}
          if(effort.effortType2Fee === null || effort.effortType2Fee === undefined || effort.effortType2Fee === '') {
            effort.effortType2Fee = 0;}
            const [result] = await connection.execute(
              `INSERT INTO UserEffortsFees (userId, userName,effortType1, effortType1Fee, effortType2, effortType2Fee)
               VALUES (?,?, ?, ?, ?, ?)`,
              [
                effort.userId,
                effort.userName,
                effort.effortType1,
                effort.effortType1Fee,
                effort.effortType2,
                effort.effortType2Fee
              ]
            );
          
        }
        
        await connection.commit();
        res.json({ success: true, message: 'Veriler başarıyla kaydedildi' });
      } catch (error) {
        await connection.rollback();
        throw error;
      } finally {
        connection.release();
      }
    } catch (error) {
      console.error('Veritabanı hatası:', error);
      res.status(500).json({ 
        success: false, 
        message: 'Veritabanı işlemi sırasında bir hata oluştu' 
      });
    }
  });
  
router.post('/AllEffortsFees', async (req, res) => {

  const {UserID} = req.body;

  const pool = await getPoolByUserId(UserID);

  try {
    const [efforts] = await pool.query(
      `SELECT * FROM UserEffortsFees`
    );
    res.status(200).json(efforts);
  } catch (error) {
    console.error('GetDailyEffort Hatası:', error);
    res.status(500).json({ error: 'Mağaza verileri alınırken bir hata oluştu.' });
  }
});

router.post('/getExpenses', async (req, res) => {
  try {
    const {UserID} = req.body;

  const pool = await getPoolByUserId(UserID);

    const [expenses] = await pool.query(
      `SELECT * FROM ShopExpenses ORDER BY startDate DESC`
    );
    res.status(200).json(expenses);
  } catch (error) {
    console.error('Error fetching expenses:', error);
    res.status(500).json({ error: 'Failed to fetch expenses' });
  }
});
  
  // Yeni gider ekle
  router.post('/expenses', async (req, res) => {
    const { UserID,shop, expenseType, startDate, endDate, amount } = req.body;
    console.log('UserID:', UserID);
     const pool = await getPoolByUserId(UserID);
    const StartDate = formatDate(startDate);
    const EndDate = formatDate(endDate);
    try {
      const query = `
        INSERT INTO ShopExpenses (shop, expenseType, startDate, endDate, amount)
        VALUES (?, ?, ?, ?, ?)
      `;
      const [result] = await pool.query(query, [shop, expenseType, StartDate, EndDate, amount]);
      res.status(201).json({ message: 'Expense added successfully', id: result.insertId });
    } catch (error) {
      console.error('Error adding expense:', error);
      res.status(500).json({ error: 'Failed to add expense' });
    }
  });
  
  // Gider güncelle
  router.put('/expenses/:id', async (req, res) => {
    const { id } = req.params;
    const { UserID,shop, expenseType, startDate, endDate, amount } = req.body;
    const pool = await getPoolByUserId(UserID);
    const StartDate = formatDate(startDate);
    const EndDate = formatDate(endDate);
    try {
      const query = `
        UPDATE ShopExpenses
        SET shop = ?, expenseType = ?, startDate = ?, endDate = ?, amount = ?
        WHERE id = ?
      `;
      await pool.query(query, [shop, expenseType, StartDate, EndDate, amount, id]);
      res.status(200).json({ message: 'Expense updated successfully' });
    } catch (error) {
      console.error('Error updating expense:', error);
      res.status(500).json({ error: 'Failed to update expense' });
    }
  });
  
  // Gider sil
  router.delete('/expenses/:id', async (req, res) => {
    const { id } = req.params;
     const { UserID } = req.body;
     const pool = await getPoolByUserId(UserID);
    try {
      const query = `DELETE FROM ShopExpenses WHERE id = ?`;
      await pool.query(query, [id]);
      res.status(200).json({ message: 'Expense deleted successfully' });
    } catch (error) {
      console.error('Error deleting expense:', error);
      res.status(500).json({ error: 'Failed to delete expense' });
    }
  });

  router.delete('/expenses', async (req, res) => {
  const { UserID,ids } = req.body;
   const pool = await getPoolByUserId(UserID);
  if (!ids || !Array.isArray(ids) || ids.length === 0) {
    return res.status(400).json({ error: 'Invalid or empty ids array' });
  }
  
  try {
    const placeholders = ids.map(() => '?').join(',');
    const query = `DELETE FROM ShopExpenses WHERE id IN (${placeholders})`;
    
    const [result] = await pool.query(query, ids);
    
    res.status(200).json({ 
      message: `${result.affectedRows} expense(s) deleted successfully`,
      deletedCount: result.affectedRows
    });
  } catch (error) {
    console.error('Error deleting expenses:', error);
    res.status(500).json({ error: 'Failed to delete expenses' });
  }
});

  router.get('/GetSecreenShotsUsers', async (req, res) => {
    try {
      const [SecreenShots] = await promisePool.query(
        `SELECT DISTINCT LocalIP, HostName, UserName FROM SecreenShots;`
      );
      res.status(200).json(SecreenShots);
    } catch (error) {
      console.error('Error fetching SecreenShots:', error);
      res.status(500).json({ error: 'Failed to fetch SecreenShots' });
    }
  });
  
  router.post('/SaveScreenShotAuth', async (req, res) => {
    const connection = await promisePool.getConnection();
  
    try {
      await connection.beginTransaction();
  
      const insertQuery = `
        INSERT INTO SecreenShotsAuth 
        (LocalIP, HostName, UserName, SystemUserID, SystemUserName) 
        VALUES (?, ?, ?, ?, ?)
      `;
  
      for (const item of req.body) {
        await connection.query(insertQuery, [
          item.LocalIP, 
          item.HostName, 
          item.UserName, 
          item.SystemUserID, 
          item.SystemUserName
        ]);
      }
  
      await connection.commit();
      res.status(200).json({ message: 'Kayıt başarılı' });
  
    } catch (error) {
      await connection.rollback();
      console.error('Kayıt hatası:', error);
      res.status(500).json({ error: 'Kayıt sırasında bir hata oluştu' });
    } finally {
      connection.release();
    }
  });

  router.get('/GetPreviousScreenShotAuth', async (req, res) => {
    try {
      const [previousAuths] = await promisePool.query(`
        SELECT LocalIP, HostName, UserName, SystemUserID, SystemUserName 
        FROM SecreenShotsAuth 
        ORDER BY CreateDate DESC
      `);
      
      res.status(200).json(previousAuths);
    } catch (error) {
      console.error('Error fetching previous authorizations:', error);
      res.status(500).json({ error: 'Failed to fetch previous authorizations' });
    }
  });

  router.post("/GetScreenShots", async (req, res) => {
    try {
        const { startDate, endDate, userId, userName } = req.body;

        console.log('startDate:', startDate);
        console.log('endDate:', endDate);
        console.log('userId:', userId);
        console.log('userName:', userName);

        // Gerekli alanların olup olmadığını kontrol et
        if (!startDate || !endDate || (!userId && !userName)) {
            return res.status(400).json({
                error: "Başlangıç tarihi, bitiş tarihi ve en az bir Kullanıcı bilgisi (ID veya isim) gereklidir."
            });
        }

        let sqlQuery = `
            SELECT DISTINCT ss.ScreenshotPath,
             ss.UploadDate, sa.SystemUserID, sa.SystemUserName, ss.Type,
             DATE_FORMAT(ss.UploadDate, '%Y-%m-%d %H:%i:%s') AS UploadDateString
            FROM SecreenShots ss
            INNER JOIN SecreenShotsAuth sa ON ss.LocalIP = sa.LocalIP
            WHERE ss.UploadDate BETWEEN ? AND ? AND ss.ScreenshotPath IS NOT NULL
        `;

        let queryParams = [startDate, endDate];

        if (userId) {
            sqlQuery += ` AND sa.SystemUserID = ?`;
            queryParams.push(userId);
        }

        if (userName) {
            sqlQuery += ` AND sa.SystemUserName = ?`;
            queryParams.push(userName);
        }

        // MySQL sorgusunu çalıştır
        const [results] = await promisePool.query(sqlQuery, queryParams);
        results.forEach(row => {
            console.log(row.UploadDateString);
        });
      
        // Yanıt olarak JSON gönder
        res.status(200).json(results);
    } catch (error) {
        console.error("GetScreenShots Hatası:", error);
        res.status(500).json({ error: "Veri alınırken bir hata oluştu." });
    }
});


router.post('/GetActivePersonnels', async (req, res) => {

   const { UserID } = req.body;
   const pool = await getPoolByUserId(UserID);

  try {
    const [Users] = await pool.query(
      `select UserID,Username from Users where Status = 1;`
    );
    res.status(200).json(Users);
  } catch (error) {
    console.error('Error fetching Users:', error);
    res.status(500).json({ error: 'Failed to fetch Users' });
  }
});
  
router.post("/GetShopData", async (req, res) => {

   const { UserID } = req.body;
   const pool = await getPoolByUserId(UserID);

  try {
    const [Shops] = await pool.query(
      `select * from Shops where status = 1`
    );
    res.status(200).json(Shops);
  } catch (error) {
    console.error("GetDailyEffort Hatası:", error);
    res
      .status(500)
      .json({ error: "Mağaza verileri alınırken bir hata oluştu." });
  }
});

router.post("/GetAllShopData", async (req, res) => {

  const { UserID } = req.body;
   const pool = await getPoolByUserId(UserID);

  try {
    const [Shops] = await pool.query(
      `select * from Shops where status = 1`
    );
    res.status(200).json(Shops);
  } catch (error) {
    console.error("GetDailyEffort Hatası:", error);
    res
      .status(500)
      .json({ error: "Mağaza verileri alınırken bir hata oluştu." });
  }
});

router.post("/InsertNewShop", async (req, res) => {
  const { ShopName, ShopID, Status,UserID } = req.body;
   const pool = await getPoolByUserId(UserID);

  let StasusBool = Status === "active" ? 1 : 0;
  try {
    const [result] = await pool.query(
      `INSERT INTO Shops (ShopName, ID, Status) VALUES (?, ?, ?)`,
      [ShopName, ShopID, StasusBool]
    );
    res.status(200).json({ message: "Mağaza başarıyla eklendi" });
  } catch (error) {
    console.error("InsertNewShop Hatası:", error);
    res.status(500).json({ error: "Mağaza eklenirken bir hata oluştu." });
  }
});

router.post("/UpdateShop", async (req, res) => {
  const { ShopName, OldShopName, ShopID, Status,UserID } = req.body;
  const pool = await getPoolByUserId(UserID);
  let StasusBool = Status === "active" ? 1 : 0;
  const connection = await pool.getConnection(); 
  try {
    await connection.beginTransaction(); 

    await connection.query(
      `UPDATE Shops SET ShopName = ?, Status = ? WHERE ID = ?`,
      [ShopName, StasusBool, ShopID]
    );

    await connection.query(
      `UPDATE ShopsAuth SET ShopName = ? WHERE ShopID = ?`,
      [ShopName, ShopID]
    );

    await connection.query(
      `UPDATE ShopFees SET SelectedShop = ? WHERE SelectedShop = ?`,
      [ShopName, OldShopName]
    );

    await connection.query(
      `UPDATE ShopExpenses SET shop = ? WHERE shop = ?`,
      [ShopName, OldShopName]
    );

    await connection.commit();
    res.status(200).json({ message: "Mağaza başarıyla güncellendi" });

  } catch (error) {
    await connection.rollback(); 
    console.error("UpdateShop Hatası:", error);
    res.status(500).json({ error: "Mağaza güncellenirken bir hata oluştu." });
  } finally {
    connection.release(); 
  }
});

router.post("/DeleteShop", async (req, res) => {
  const {  ShopID, UserID} = req.body;
   const pool = await getPoolByUserId(UserID);

  const connection = await pool.getConnection(); 
  try {
    await connection.beginTransaction(); 

    await connection.query(
      `DELETE FROM Shops WHERE ID = ?`,
      [ShopID]
    );
    res.status(200).json({ message: "Mağaza başarıyla Silindi" });

  } catch (error) {
    await connection.rollback(); 
    console.error("UpdateShop Hatası:", error);
    res.status(500).json({ error: "Mağaza güncellenirken bir hata oluştu." });
  } finally {
    connection.release(); 
  }
});

router.post("/getShopsByCaptainOrSelf", async (req, res) => {
  const { UserID, IsAdmin } = req.body;
  console.log('getShopsByCaptainOrSelf', UserID, IsAdmin);
  if (!UserID) {
    return res.status(400).json({ error: "UserID gerekli." });
  }

  try {
    const pool = await getPoolByUserId(UserID);

    let results;

    if (IsAdmin === 1 || IsAdmin === "1") {
      // Admin ise tüm kayıtları getir
      [results] = await pool.query(`SELECT DISTINCT UserID,ShopName,ShopID FROM ShopsAuth`);
    } else {
      // Değilse sadece kendi ve kaptanı olduğu kullanıcıların verileri
      [results] = await pool.query(
        `
        SELECT UserID,ShopName,ShopID FROM ShopsAuth
        WHERE UserID IN (
          SELECT StaffUserID FROM UserAuth WHERE CaptainUserID = ?
        )
        OR UserID = ?
        `,
        [UserID, UserID]
      );
    }

    res.status(200).json(results);
  } catch (error) {
    console.error("getShopsByCaptainOrSelf Hatası:", error);
    res.status(500).json({ error: "Veriler alınırken bir hata oluştu." });
  }
});



module.exports = router;
