const express = require("express");
const fs = require("fs");
const multer = require("multer");
const mysql = require("mysql2");
const xlsx = require("xlsx");
const path = require("path");
const router = express.Router();
const app = express();

app.use(express.json());
const { promisePool } = require("./db");

// Multer configuration for file upload
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = './uploads/';
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const timestamp = Date.now();
    const originalName = Buffer.from(file.originalname, 'latin1').toString('utf8');
    cb(null, `${timestamp}-${originalName}`);
  }
});

const upload = multer({ 
  storage: storage,
  fileFilter: (req, file, cb) => {
    const allowedTypes = ['.xlsx', '.xls'];
    const fileExtension = path.extname(file.originalname).toLowerCase();
    
    if (allowedTypes.includes(fileExtension)) {
      cb(null, true);
    } else {
      cb(new Error('Sadece Excel dosyaları (.xlsx, .xls) yüklenebilir!'));
    }
  },
  limits: {
    fileSize: 50 * 1024 * 1024 // 50MB limit
  }
});

// Excel tarih formatını MySQL tarih formatına çevir
const formatExcelDateToMySQL = (excelDate) => {
  if (!excelDate) return null;
  
  try {
    // Excel'den gelen numeric tarih değerini kontrol et
    if (typeof excelDate === 'number') {
      const date = new Date((excelDate - 25569) * 86400 * 1000);
      return date.toISOString().split('T')[0];
    }
    
    // String tarih formatını kontrol et
    if (typeof excelDate === 'string') {
      const date = new Date(excelDate);
      if (!isNaN(date.getTime())) {
        return date.toISOString().split('T')[0];
      }
    }
    
    return null;
  } catch (error) {
    console.error('Tarih formatı hatası:', error);
    return null;
  }
};

// Veri temizleme fonksiyonu
const cleanData = (value) => {
  if (value === null || value === undefined || value === '') return null;
  if (typeof value === 'string') {
    return value.trim();
  }
  return value;
};

// ALIŞ Excel dosyası işleme
router.post("/upload-alis", upload.single('alisFile'), async (req, res) => {
  let connection;
  
  try {
    if (!req.file) {
      return res.status(400).json({
        success: false,
        message: "Alış Excel dosyası yüklenmedi!"
      });
    }

    console.log("Alış dosyası yüklendi:", req.file.filename);

    // Excel dosyasını oku
    const workbook = xlsx.readFile(req.file.path);
    
    if (!workbook.SheetNames.includes('DATA')) {
      return res.status(400).json({
        success: false,
        message: "Excel dosyasında 'DATA' sheet'i bulunamadı!"
      });
    }

    const worksheet = workbook.Sheets['DATA'];
    const jsonData = xlsx.utils.sheet_to_json(worksheet, { header: 1 });

    if (jsonData.length < 2) {
      return res.status(400).json({
        success: false,
        message: "Excel dosyasında yeterli veri bulunamadı!"
      });
    }

    // Header kontrolü
    const headers = jsonData[0];
    const requiredHeaders = ['VENDOR ', 'TARİH', 'ÜRÜN SKU', 'COLOR', 'SİZE', 'QTY', 'Price', 'BİRLEŞİK İSİM'];
    const missingHeaders = requiredHeaders.filter(header => !headers.includes(header));
    
    if (missingHeaders.length > 0) {
      return res.status(400).json({
        success: false,
        message: `Eksik sütunlar: ${missingHeaders.join(', ')}`
      });
    }

    // Transaction başlat
    connection = await promisePool.getConnection();
    await connection.beginTransaction();

    let successCount = 0;
    let errorCount = 0;
    const errors = [];

    // Batch insert için veri hazırla
    const batchSize = 500;
    const insertData = [];

    for (let i = 1; i < jsonData.length; i++) {
      try {
        const row = jsonData[i];
        
        // Boş satırları atla
        if (!row || row.every(cell => !cell)) continue;

        const vendor = cleanData(row[0]);
        const tarih = formatExcelDateToMySQL(row[1]);
        const faturaNo = cleanData(row[2]);
        const urunSku = cleanData(row[3]);
        const aciklama = cleanData(row[4]);
        const renk = cleanData(row[5]);
        const beden = cleanData(row[6]);
        const miktar = parseInt(row[7]) || 0;
        const birimFiyat = parseFloat(row[8]) || 0;
        const toplamFiyat = parseFloat(row[10]) || 0;
        const birlesikIsim = cleanData(row[11]);
        const marka = cleanData(row[16]);
        const tshirtType = cleanData(row[17]);

        // Zorunlu alanları kontrol et
        if (!vendor || !birlesikIsim || miktar <= 0) {
          errorCount++;
          errors.push(`Satır ${i + 1}: Eksik zorunlu alan (vendor: ${vendor}, birlesik_isim: ${birlesikIsim}, miktar: ${miktar})`);
          continue;
        }

        insertData.push([
          vendor,
          tarih,
          faturaNo,
          urunSku,
          aciklama,
          renk,
          beden,
          miktar,
          birimFiyat,
          toplamFiyat,
          birlesikIsim,
          marka,
          tshirtType
        ]);

        successCount++;

      } catch (rowError) {
        errorCount++;
        errors.push(`Satır ${i + 1}: ${rowError.message}`);
      }
    }

    if (insertData.length === 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: "İşlenebilir veri bulunamadı!",
        errors: errors.slice(0, 10) // İlk 10 hatayı göster
      });
    }

    // Batch insert işlemi
    const insertQuery = `
      INSERT INTO alis_hareketleri 
      (vendor, tarih, fatura_no, urun_sku, aciklama, renk, beden, miktar, birim_fiyat, toplam_fiyat, birlesik_isim, marka, tshirt_type)
      VALUES ?
      ON DUPLICATE KEY UPDATE
      miktar = VALUES(miktar),
      birim_fiyat = VALUES(birim_fiyat),
      toplam_fiyat = VALUES(toplam_fiyat),
      kayit_tarihi = CURRENT_TIMESTAMP
    `;

    // Batch'ler halinde insert et
    for (let i = 0; i < insertData.length; i += batchSize) {
      const batch = insertData.slice(i, i + batchSize);
      await connection.query(insertQuery, [batch]);
    }

    await connection.commit();

    // Geçici dosyayı sil
    fs.unlinkSync(req.file.path);

    res.json({
      success: true,
      message: "Alış verileri başarıyla yüklendi!",
      stats: {
        totalProcessed: jsonData.length - 1,
        successCount,
        errorCount,
        duplicateHandled: true
      },
      errors: errors.slice(0, 5) // İlk 5 hatayı göster
    });

  } catch (error) {
    if (connection) {
      await connection.rollback();
    }
    
    console.error("Alış dosyası yükleme hatası:", error);
    
    // Geçici dosyayı sil
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }

    res.status(500).json({
      success: false,
      message: "Alış verileri yüklenirken hata oluştu!",
      error: error.message
    });
  } finally {
    if (connection) connection.release();
  }
});

// SATIŞ Excel dosyası işleme
router.post("/upload-satis", upload.single('satisFile'), async (req, res) => {
  let connection;
  
  try {
    if (!req.file) {
      return res.status(400).json({
        success: false,
        message: "Satış Excel dosyası yüklenmedi!"
      });
    }

    console.log("Satış dosyası yüklendi:", req.file.filename);

    // Excel dosyasını oku
    const workbook = xlsx.readFile(req.file.path);
    
    if (!workbook.SheetNames.includes('DATA')) {
      return res.status(400).json({
        success: false,
        message: "Excel dosyasında 'DATA' sheet'i bulunamadı!"
      });
    }

    const worksheet = workbook.Sheets['DATA'];
    const jsonData = xlsx.utils.sheet_to_json(worksheet, { header: 1 });

    if (jsonData.length < 2) {
      return res.status(400).json({
        success: false,
        message: "Excel dosyasında yeterli veri bulunamadı!"
      });
    }

    // Header kontrolü
    const headers = jsonData[0];
    const requiredHeaders = ['Sipariş Numarası', 'Mağaza', 'Sipariş Tarihi', 'Stok Kodu', 'Miktar', 'KOD'];
    const missingHeaders = requiredHeaders.filter(header => !headers.includes(header));
    
    if (missingHeaders.length > 0) {
      return res.status(400).json({
        success: false,
        message: `Eksik sütunlar: ${missingHeaders.join(', ')}`
      });
    }

    // Transaction başlat
    connection = await promisePool.getConnection();
    await connection.beginTransaction();

    let successCount = 0;
    let errorCount = 0;
    const errors = [];
    const batchSize = 500;
    const insertData = [];

    for (let i = 1; i < jsonData.length; i++) {
      try {
        const row = jsonData[i];
        
        // Boş satırları atla
        if (!row || row.every(cell => !cell)) continue;

        const urunAdi = cleanData(row[0]);
        const siparisNo = cleanData(row[1]);
        const magaza = cleanData(row[2]);
        const siparisTarihi = formatExcelDateToMySQL(row[3]);
        const onayTarihi = formatExcelDateToMySQL(row[4]);
        const siparisTipi = cleanData(row[5]);
        const stokKodu = cleanData(row[6]);
        const miktar = parseInt(row[7]) || 0;
        const maliyet = parseFloat(row[8]) || 0;
        const secenek1 = cleanData(row[9]);
        const secenek2 = cleanData(row[10]);
        const kod = cleanData(row[11]);

        // Zorunlu alanları kontrol et
        if (!siparisNo || !kod || miktar <= 0) {
          errorCount++;
          errors.push(`Satır ${i + 1}: Eksik zorunlu alan (siparis_no: ${siparisNo}, kod: ${kod}, miktar: ${miktar})`);
          continue;
        }

        insertData.push([
          urunAdi,
          siparisNo,
          magaza,
          siparisTarihi,
          onayTarihi,
          siparisTipi,
          stokKodu,
          miktar,
          maliyet,
          secenek1,
          secenek2,
          kod
        ]);

        successCount++;

      } catch (rowError) {
        errorCount++;
        errors.push(`Satır ${i + 1}: ${rowError.message}`);
      }
    }

    if (insertData.length === 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: "İşlenebilir veri bulunamadı!",
        errors: errors.slice(0, 10)
      });
    }

    // Batch insert işlemi
    const insertQuery = `
      INSERT INTO satis_hareketleri 
      (urun_adi, siparis_no, magaza, siparis_tarihi, onay_tarihi, siparis_tipi, stok_kodu, miktar, maliyet, secenek_1, secenek_2, kod)
      VALUES ?
      ON DUPLICATE KEY UPDATE
      miktar = VALUES(miktar),
      maliyet = VALUES(maliyet),
      kayit_tarihi = CURRENT_TIMESTAMP
    `;

    // Batch'ler halinde insert et
    for (let i = 0; i < insertData.length; i += batchSize) {
      const batch = insertData.slice(i, i + batchSize);
      await connection.query(insertQuery, [batch]);
    }

    await connection.commit();

    // Geçici dosyayı sil
    fs.unlinkSync(req.file.path);

    res.json({
      success: true,
      message: "Satış verileri başarıyla yüklendi!",
      stats: {
        totalProcessed: jsonData.length - 1,
        successCount,
        errorCount,
        duplicateHandled: true
      },
      errors: errors.slice(0, 5)
    });

  } catch (error) {
    if (connection) {
      await connection.rollback();
    }
    
    console.error("Satış dosyası yükleme hatası:", error);
    
    // Geçici dosyayı sil
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }

    res.status(500).json({
      success: false,
      message: "Satış verileri yüklenirken hata oluştu!",
      error: error.message
    });
  } finally {
    if (connection) connection.release();
  }
});

// Stok durumunu hesapla ve güncelle
router.post("/hesapla-stok", async (req, res) => {
  let connection;
  
  try {
    console.log('Stok hesaplaması başlatıldı...');
    connection = await promisePool.getConnection();
    await connection.beginTransaction();

    // Stok durumu tablosunu temizleme - KALDIRILDI
    // await connection.query("DELETE FROM stok_durumu"); // ← Bunu kaldırdık

    // Alış toplamları
    console.log('Alış toplamları hesaplanıyor...');
    const alisQuery = `
      SELECT 
        birlesik_isim as urun_kodu,
        SUM(miktar) as toplam_alis
      FROM alis_hareketleri 
      WHERE birlesik_isim IS NOT NULL AND birlesik_isim != ''
      GROUP BY birlesik_isim
    `;

    // Satış toplamları
    console.log('Satış toplamları hesaplanıyor...');
    const satisQuery = `
      SELECT 
        kod as urun_kodu,
        SUM(miktar) as toplam_satis
      FROM satis_hareketleri 
      WHERE kod IS NOT NULL AND kod != ''
      GROUP BY kod
    `;

    const [alisResults] = await connection.query(alisQuery);
    const [satisResults] = await connection.query(satisQuery);

    console.log(`Alış kayıtları: ${alisResults.length}, Satış kayıtları: ${satisResults.length}`);

    // Tüm ürün kodlarını topla
    const allProducts = new Set();
    alisResults.forEach(row => allProducts.add(row.urun_kodu));
    satisResults.forEach(row => allProducts.add(row.urun_kodu));

    console.log(`Toplam benzersiz ürün: ${allProducts.size}`);

    // UPSERT yaklaşımı - tek tek veya batch ile
    let processedCount = 0;
    let updatedCount = 0;
    let insertedCount = 0;

    // Batch UPSERT için veri hazırla
    const stokInsertData = [];
    
    for (const urunKodu of allProducts) {
      if (!urunKodu || urunKodu.trim() === '') continue;

      const alisRow = alisResults.find(row => row.urun_kodu === urunKodu);
      const satisRow = satisResults.find(row => row.urun_kodu === urunKodu);
      
      const toplamAlis = alisRow ? alisRow.toplam_alis : 0;
      const toplamSatis = satisRow ? satisRow.toplam_satis : 0;
      const guncelStok = toplamAlis - toplamSatis;
      
      stokInsertData.push([urunKodu.trim(), toplamAlis, toplamSatis, guncelStok]);
      processedCount++;
    }

    if (stokInsertData.length > 0) {
      console.log(`${stokInsertData.length} ürün için stok durumu kaydediliyor...`);
      
      // UPSERT query - ON DUPLICATE KEY UPDATE ile
      const insertStokQuery = `
        INSERT INTO stok_durumu (urun_kodu, toplam_alis, toplam_satis, guncel_stok)
        VALUES ?
        ON DUPLICATE KEY UPDATE
        toplam_alis = VALUES(toplam_alis),
        toplam_satis = VALUES(toplam_satis),
        guncel_stok = VALUES(guncel_stok),
        son_guncelleme = CURRENT_TIMESTAMP
      `;
      
      const [result] = await connection.query(insertStokQuery, [stokInsertData]);
      
      // Kaç kayıt eklenip kaç kayıt güncellendiğini hesapla
      insertedCount = result.affectedRows - result.changedRows;
      updatedCount = result.changedRows;
    }

    await connection.commit();
    console.log(`Stok hesaplaması tamamlandı! İşlenen: ${processedCount}, Yeni: ${insertedCount}, Güncellenen: ${updatedCount}`);

    // Özet bilgileri hazırla
    const ozetQuery = `
      SELECT 
        COUNT(*) as toplam_urun,
        SUM(CASE WHEN guncel_stok > 0 THEN 1 ELSE 0 END) as stokta_olan,
        SUM(CASE WHEN guncel_stok = 0 THEN 1 ELSE 0 END) as stoku_biten,
        SUM(CASE WHEN guncel_stok < 0 THEN 1 ELSE 0 END) as eksi_stoklu
      FROM stok_durumu
    `;
    
    const [ozetResult] = await connection.query(ozetQuery);

    res.json({
      success: true,
      message: "Stok durumu başarıyla hesaplandı!",
      stats: {
        totalProducts: allProducts.size,
        processedProducts: processedCount,
        insertedProducts: insertedCount,
        updatedProducts: updatedCount,
        alisKayitlari: alisResults.length,
        satisKayitlari: satisResults.length,
        ozet: ozetResult[0]
      }
    });

  } catch (error) {
    if (connection) {
      await connection.rollback();
    }
    
    console.error("Stok hesaplama hatası:", error);
    res.status(500).json({
      success: false,
      message: "Stok hesaplama işlemi başarısız!",
      error: error.message
    });
  } finally {
    if (connection) connection.release();
  }
});
// Stok durumunu listele
router.get("/stok-durumu", async (req, res) => {
  try {
    const { page = 1, limit = 50, search = '', stokDurumu = 'all' } = req.query;
    const offset = (parseInt(page) - 1) * parseInt(limit);

    let whereClause = "WHERE 1=1";
    let queryParams = [];

    if (search) {
      whereClause += " AND urun_kodu LIKE ?";
      queryParams.push(`%${search}%`);
    }

    if (stokDurumu === 'pozitif') {
      whereClause += " AND guncel_stok > 0";
    } else if (stokDurumu === 'sifir') {
      whereClause += " AND guncel_stok = 0";
    } else if (stokDurumu === 'negatif') {
      whereClause += " AND guncel_stok < 0";
    }

    const countQuery = `SELECT COUNT(*) as total FROM stok_durumu ${whereClause}`;
    const dataQuery = `
      SELECT 
        urun_kodu,
        toplam_alis,
        toplam_satis,
        guncel_stok,
        son_guncelleme
      FROM stok_durumu 
      ${whereClause}
      ORDER BY guncel_stok DESC
      LIMIT ? OFFSET ?
    `;

    queryParams.push(parseInt(limit), offset);

    const [countResult] = await promisePool.query(countQuery, queryParams.slice(0, -2));
    const [dataResult] = await promisePool.query(dataQuery, queryParams);

    res.json({
      success: true,
      data: dataResult,
      pagination: {
        total: countResult[0].total,
        page: parseInt(page),
        limit: parseInt(limit),
        totalPages: Math.ceil(countResult[0].total / parseInt(limit))
      }
    });

  } catch (error) {
    console.error("Stok durumu listeleme hatası:", error);
    res.status(500).json({
      success: false,
      message: "Stok durumu listelenemedi!",
      error: error.message
    });
  }
});

// Error handling middleware
router.use((error, req, res, next) => {
  if (error instanceof multer.MulterError) {
    if (error.code === 'LIMIT_FILE_SIZE') {
      return res.status(400).json({
        success: false,
        message: 'Dosya boyutu çok büyük! Maksimum 50MB olmalıdır.'
      });
    }
  }
  
  res.status(500).json({
    success: false,
    message: error.message || 'Bilinmeyen bir hata oluştu!'
  });
});

// Stok Rapor API'ları - router'a eklenecek

router.get("/rapor/genel-ozet", async (req, res) => {
  try {
    const [stokOzeti] = await promisePool.query(`
      SELECT 
        COUNT(*) as toplam_urun_sayisi,
        SUM(toplam_alis) as toplam_alis_miktari,
        SUM(toplam_satis) as toplam_satis_miktari,
        SUM(guncel_stok) as toplam_guncel_stok,
        SUM(CASE WHEN guncel_stok > 0 THEN 1 ELSE 0 END) as stokta_olan_urun,
        SUM(CASE WHEN guncel_stok = 0 THEN 1 ELSE 0 END) as stoku_biten_urun,
        SUM(CASE WHEN guncel_stok < 0 THEN 1 ELSE 0 END) as eksi_stoklu_urun,
        SUM(CASE WHEN guncel_stok > 0 AND guncel_stok < 50 THEN 1 ELSE 0 END) as kritik_stoklu_urun
      FROM stok_durumu
    `);

    res.json({
      success: true,
      data: stokOzeti[0]
    });

  } catch (error) {
    console.error("Genel özet hatası:", error);
    res.status(500).json({
      success: false,
      message: "Genel özet alınamadı!",
      error: error.message
    });
  }
});

// 2. Detaylı Stok Raporu (Filtreleme + Pagination)
router.get("/rapor/detayli-stok", async (req, res) => {
  try {
    const { 
      page = 1, 
      limit = 20, 
      search = '', 
      stokDurumu = 'all',
      marka = '',
      renk = '',
      beden = '',
      sortBy = 'guncel_stok',
      sortOrder = 'DESC'
    } = req.query;

    const offset = (parseInt(page) - 1) * parseInt(limit);

    let whereClause = "WHERE 1=1";
    let queryParams = [];

    // Arama filtresi
    if (search) {
      whereClause += " AND sd.urun_kodu LIKE ?";
      queryParams.push(`%${search}%`);
    }

    // Stok durumu filtresi - Kritik stok eklendi
    if (stokDurumu === 'pozitif') {
      whereClause += " AND sd.guncel_stok > 0";
    } else if (stokDurumu === 'sifir') {
      whereClause += " AND sd.guncel_stok = 0";
    } else if (stokDurumu === 'negatif') {
      whereClause += " AND sd.guncel_stok < 0";
    } else if (stokDurumu === 'kritik') {
      whereClause += " AND sd.guncel_stok > 0 AND sd.guncel_stok < 50";
    }

    // Marka filtresi
    if (marka) {
      whereClause += " AND ah.marka LIKE ?";
      queryParams.push(`%${marka}%`);
    }

    // Renk filtresi
    if (renk) {
      whereClause += " AND ah.renk LIKE ?";
      queryParams.push(`%${renk}%`);
    }

    // Beden filtresi
    if (beden) {
      whereClause += " AND ah.beden LIKE ?";
      queryParams.push(`%${beden}%`);
    }

    // Sıralama kontrolü
    const allowedSortFields = ['guncel_stok', 'toplam_alis', 'toplam_satis', 'urun_kodu'];
    const sortField = allowedSortFields.includes(sortBy) ? sortBy : 'guncel_stok';
    const sortDirection = sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';

    // Count query - Subquery ile
    const countQuery = `
      SELECT COUNT(*) as total 
      FROM stok_durumu sd
      LEFT JOIN (
        SELECT DISTINCT birlesik_isim, marka, renk, beden
        FROM alis_hareketleri
      ) ah_data ON sd.urun_kodu = ah_data.birlesik_isim
      ${whereClause.replace('ah.', 'ah_data.')}
    `;

    // Data query - Subquery ile daha temiz çözüm
    const dataQuery = `
      SELECT 
        sd.urun_kodu,
        sd.toplam_alis,
        sd.toplam_satis,
        sd.guncel_stok,
        sd.son_guncelleme,
        COALESCE(ah_data.marka, '') as marka,
        COALESCE(ah_data.renk, '') as renk,
        COALESCE(ah_data.beden, '') as beden,
        COALESCE(ah_data.tshirt_type, '') as tshirt_type,
        COALESCE(ah_data.birim_fiyat, 0) as son_alis_fiyati,
        CASE 
          WHEN sd.guncel_stok > 10 THEN 'Yeterli'
          WHEN sd.guncel_stok > 0 THEN 'Az'
          WHEN sd.guncel_stok = 0 THEN 'Tükendi'
          ELSE 'Eksik'
        END as stok_durumu_text
      FROM stok_durumu sd
      LEFT JOIN (
        SELECT 
          birlesik_isim,
          MAX(marka) as marka,
          MAX(renk) as renk,
          MAX(beden) as beden,
          MAX(tshirt_type) as tshirt_type,
          MAX(birim_fiyat) as birim_fiyat
        FROM alis_hareketleri
        GROUP BY birlesik_isim
      ) ah_data ON sd.urun_kodu = ah_data.birlesik_isim
      ${whereClause.replace('ah.', 'ah_data.')}
      ORDER BY sd.${sortField} ${sortDirection}
      LIMIT ? OFFSET ?
    `;

    queryParams.push(parseInt(limit), offset);

    const [countResult] = await promisePool.query(countQuery, queryParams.slice(0, -2));
    const [dataResult] = await promisePool.query(dataQuery, queryParams);

    res.json({
      success: true,
      data: dataResult,
      pagination: {
        total: countResult[0].total,
        page: parseInt(page),
        limit: parseInt(limit),
        totalPages: Math.ceil(countResult[0].total / parseInt(limit))
      }
    });

  } catch (error) {
    console.error("Detaylı stok raporu hatası:", error);
    res.status(500).json({
      success: false,
      message: "Detaylı stok raporu alınamadı!",
      error: error.message
    });
  }
});

// 3. Beden Bazında Stok Analizi
router.get("/rapor/beden-analizi", async (req, res) => {
  try {
    const [bedenAnalizi] = await promisePool.query(`
      SELECT 
        ah.beden,
        COUNT(DISTINCT sd.urun_kodu) as urun_sayisi,
        SUM(sd.toplam_alis) as toplam_alis,
        SUM(sd.toplam_satis) as toplam_satis,
        SUM(sd.guncel_stok) as toplam_stok,
        AVG(sd.guncel_stok) as ortalama_stok,
        SUM(CASE WHEN sd.guncel_stok > 0 THEN 1 ELSE 0 END) as stokta_olan,
        SUM(CASE WHEN sd.guncel_stok = 0 THEN 1 ELSE 0 END) as tukenen
      FROM stok_durumu sd
      LEFT JOIN alis_hareketleri ah ON sd.urun_kodu = ah.birlesik_isim
      WHERE ah.beden IS NOT NULL
      GROUP BY ah.beden
      ORDER BY toplam_stok DESC
    `);

    res.json({
      success: true,
      data: bedenAnalizi
    });

  } catch (error) {
    console.error("Beden analizi hatası:", error);
    res.status(500).json({
      success: false,
      message: "Beden analizi alınamadı!",
      error: error.message
    });
  }
});

// 4. Renk Bazında Stok Analizi - GROUP BY düzeltmesi
router.get("/rapor/renk-analizi", async (req, res) => {
  try {
    const [renkAnalizi] = await promisePool.query(`
      SELECT 
        ah_summary.renk,
        COUNT(DISTINCT sd.urun_kodu) as urun_sayisi,
        SUM(sd.toplam_alis) as toplam_alis,
        SUM(sd.toplam_satis) as toplam_satis,
        SUM(sd.guncel_stok) as toplam_stok,
        ROUND(SUM(sd.toplam_satis) * 100.0 / NULLIF(SUM(sd.toplam_alis), 0), 2) as satis_orani
      FROM stok_durumu sd
      LEFT JOIN (
        SELECT birlesik_isim, MAX(renk) as renk
        FROM alis_hareketleri
        WHERE renk IS NOT NULL
        GROUP BY birlesik_isim
      ) ah_summary ON sd.urun_kodu = ah_summary.birlesik_isim
      WHERE ah_summary.renk IS NOT NULL
      GROUP BY ah_summary.renk
      ORDER BY toplam_stok DESC
      LIMIT 20
    `);

    res.json({
      success: true,
      data: renkAnalizi
    });

  } catch (error) {
    console.error("Renk analizi hatası:", error);
    res.status(500).json({
      success: false,
      message: "Renk analizi alınamadı!",
      error: error.message
    });
  }
});

// 5. Marka Bazında Analiz
router.get("/rapor/marka-analizi", async (req, res) => {
  try {
    const [markaAnalizi] = await promisePool.query(`
      SELECT 
        ah.marka,
        COUNT(DISTINCT sd.urun_kodu) as urun_sayisi,
        SUM(sd.toplam_alis) as toplam_alis,
        SUM(sd.toplam_satis) as toplam_satis,
        SUM(sd.guncel_stok) as toplam_stok,
        SUM(sd.toplam_alis * ah.birim_fiyat) as toplam_alis_tutari,
        ROUND(AVG(ah.birim_fiyat), 2) as ortalama_birim_fiyat
      FROM stok_durumu sd
      LEFT JOIN alis_hareketleri ah ON sd.urun_kodu = ah.birlesik_isim
      WHERE ah.marka IS NOT NULL
      GROUP BY ah.marka
      ORDER BY toplam_stok DESC
    `);

    res.json({
      success: true,
      data: markaAnalizi
    });

  } catch (error) {
    console.error("Marka analizi hatası:", error);
    res.status(500).json({
      success: false,
      message: "Marka analizi alınamadı!",
      error: error.message
    });
  }
});

// 6. Kritik Stok Uyarıları
router.get("/rapor/kritik-stok", async (req, res) => {
  try {
    const { minStok = 5 } = req.query;

    const [kritikStok] = await promisePool.query(`
      SELECT 
        sd.urun_kodu,
        sd.guncel_stok,
        ah.marka,
        ah.renk,
        ah.beden,
        ah.birim_fiyat,
        sd.toplam_satis,
        CASE 
          WHEN sd.guncel_stok < 0 THEN 'Eksik Stok'
          WHEN sd.guncel_stok = 0 THEN 'Stok Tükendi'
          WHEN sd.guncel_stok <= ? THEN 'Kritik Seviye'
          ELSE 'Normal'
        END as uyari_tipi,
        ROUND(sd.toplam_satis / 30.0, 2) as gunluk_ortalama_satis
      FROM stok_durumu sd
      LEFT JOIN alis_hareketleri ah ON sd.urun_kodu = ah.birlesik_isim
      WHERE sd.guncel_stok <= ?
      ORDER BY sd.guncel_stok ASC, sd.toplam_satis DESC
      LIMIT 50
    `, [minStok, minStok]);

    res.json({
      success: true,
      data: kritikStok
    });

  } catch (error) {
    console.error("Kritik stok uyarıları hatası:", error);
    res.status(500).json({
      success: false,
      message: "Kritik stok uyarıları alınamadı!",
      error: error.message
    });
  }
});

// 7. En Çok Satan Ürünler
router.get("/rapor/en-cok-satan", async (req, res) => {
  try {
    const { limit = 20 } = req.query;

    const [enCokSatan] = await promisePool.query(`
      SELECT 
        sd.urun_kodu,
        sd.toplam_satis,
        sd.toplam_alis,
        sd.guncel_stok,
        ah.marka,
        ah.renk,
        ah.beden,
        ah.birim_fiyat,
        ROUND(sd.toplam_satis * 100.0 / NULLIF(sd.toplam_alis, 0), 2) as satis_orani,
        sd.toplam_satis * ah.birim_fiyat as tahmini_ciro
      FROM stok_durumu sd
      LEFT JOIN alis_hareketleri ah ON sd.urun_kodu = ah.birlesik_isim
      WHERE sd.toplam_satis > 0
      ORDER BY sd.toplam_satis DESC
      LIMIT ?
    `, [parseInt(limit)]);

    res.json({
      success: true,
      data: enCokSatan
    });

  } catch (error) {
    console.error("En çok satan ürünler hatası:", error);
    res.status(500).json({
      success: false,
      message: "En çok satan ürünler alınamadı!",
      error: error.message
    });
  }
});

// 8. Filtre Seçeneklerini Getir
router.get("/rapor/filtre-secenekleri", async (req, res) => {
  try {
    const [markalar] = await promisePool.query(`
      SELECT DISTINCT marka 
      FROM alis_hareketleri 
      WHERE marka IS NOT NULL AND marka != '' 
      ORDER BY marka
    `);

    const [renkler] = await promisePool.query(`
      SELECT DISTINCT renk 
      FROM alis_hareketleri 
      WHERE renk IS NOT NULL AND renk != '' 
      ORDER BY renk
    `);

    const [bedenler] = await promisePool.query(`
      SELECT DISTINCT beden 
      FROM alis_hareketleri 
      WHERE beden IS NOT NULL AND beden != '' 
      ORDER BY 
        CASE 
          WHEN beden = 'XS' THEN 1
          WHEN beden = 'S' THEN 2
          WHEN beden = 'M' THEN 3
          WHEN beden = 'L' THEN 4
          WHEN beden = 'XL' THEN 5
          WHEN beden = '2XL' THEN 6
          WHEN beden = '3XL' THEN 7
          ELSE 8
        END
    `);

    res.json({
      success: true,
      data: {
        markalar: markalar.map(m => m.marka),
        renkler: renkler.map(r => r.renk),
        bedenler: bedenler.map(b => b.beden)
      }
    });

  } catch (error) {
    console.error("Filtre seçenekleri hatası:", error);
    res.status(500).json({
      success: false,
      message: "Filtre seçenekleri alınamadı!",
      error: error.message
    });
  }
});

// 9. Excel Export için veri
router.get("/rapor/excel-export", async (req, res) => {
  try {
    const [exportData] = await promisePool.query(`
      SELECT 
        sd.urun_kodu as 'Ürün Kodu',
        ah.marka as 'Marka',
        ah.renk as 'Renk',
        ah.beden as 'Beden',
        sd.toplam_alis as 'Toplam Alış',
        sd.toplam_satis as 'Toplam Satış',
        sd.guncel_stok as 'Güncel Stok',
        ah.birim_fiyat as 'Birim Fiyat',
        CASE 
          WHEN sd.guncel_stok > 10 THEN 'Yeterli'
          WHEN sd.guncel_stok > 0 THEN 'Az'
          WHEN sd.guncel_stok = 0 THEN 'Tükendi'
          ELSE 'Eksik'
        END as 'Stok Durumu',
        sd.son_guncelleme as 'Son Güncelleme'
      FROM stok_durumu sd
      LEFT JOIN alis_hareketleri ah ON sd.urun_kodu = ah.birlesik_isim
      ORDER BY sd.guncel_stok DESC
    `);

    res.json({
      success: true,
      data: exportData
    });

  } catch (error) {
    console.error("Excel export hatası:", error);
    res.status(500).json({
      success: false,
      message: "Excel export verisi alınamadı!",
      error: error.message
    });
  }
});

module.exports = router;