const express = require("express");
const fs = require("fs");
const multer = require("multer");
const mysql = require("mysql2");
const xlsx = require("xlsx");
const path = require("path");
const router = express.Router();
const app = express();
app.use(express.json());

const { promisePool } = require("./db");
const { getPoolByUserId } = require("./databaseSelector");

router.get('/squRecords', async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 50;
    const offset = (page - 1) * limit;
console.log('squRecords GET:', page, limit, offset);
    const [countResult] = await promisePool.execute('SELECT COUNT(*) as total FROM SquMaintenanceScreen');
    const total = countResult[0].total;

    // SQL'de direkt değerleri kullan
    const [results] = await promisePool.execute(
      `SELECT * FROM SquMaintenanceScreen ORDER BY id DESC LIMIT ${limit} OFFSET ${offset}`
    );

    res.json({ total, records: results });
  } catch (err) {
    console.error('Kayıtlar getirilemedi:', err);
    return res.status(500).json({ error: 'Kayıtlar getirilemedi' });
  }
});


// Yeni kayıt ekle
router.post('/squRecords', async (req, res) => {
  const { SquName, creatorUser, shopName } = req.body;  
  console.log('squRecords POST:', SquName, creatorUser, shopName);
  if (!SquName || !creatorUser) {
    return res.status(400).json({ error: 'SquName ve creatorUser gerekli' });
  }

  try {
    // Son SquID'yi getir
    const getLastSquIDQuery = 'SELECT SquID FROM SquMaintenanceScreen ORDER BY id DESC LIMIT 1';
    const [results] = await promisePool.execute(getLastSquIDQuery);

    let newSquID;
    if (results.length === 0) {
      // İlk kayıt
      newSquID = '010';
    } else {
      // Son SquID'den sayısal kısmı çıkar ve +10 ekle
      const lastSquID = results[0].SquID;
      const lastNumber = parseInt(lastSquID);
      const newNumber = lastNumber + 10;
      newSquID = newNumber.toString(); 
    }

    // Yeni kaydı ekle
    const insertQuery = 'INSERT INTO SquMaintenanceScreen (SquID, SquName, creatorUser,shopName) VALUES (?, ?, ?, ?)';
    const [result] = await promisePool.execute(insertQuery, [newSquID, SquName, creatorUser, shopName]);
    
    res.status(201).json({
      message: 'Kayıt başarıyla eklendi',
      id: result.insertId,
      SquID: newSquID
    });
  } catch (err) {
    console.error('Kayıt eklenemedi:', err);
    return res.status(500).json({ error: 'Kayıt eklenemedi' });
  }
});

router.put('/squRecords/:id', async (req, res) => {
  const { id } = req.params;
  const { SquName, creatorUser, userID, isAdmin, shopName } = req.body;
  
  if (!SquName || !creatorUser) {
    return res.status(400).json({ error: 'SquName ve creatorUser gerekli' });
  }

  try {
    // Önce kaydın sahibini kontrol et
    const checkQuery = 'SELECT creatorUser FROM SquMaintenanceScreen WHERE id = ?';
    const [checkResult] = await promisePool.execute(checkQuery, [id]);
    
    if (checkResult.length === 0) {
      return res.status(404).json({ error: 'Kayıt bulunamadı' });
    }

    const recordOwner = checkResult[0].creatorUser;
    
    // Yetki kontrolü
    let hasPermission = false;
    
    if (isAdmin) {
      hasPermission = true;
    } else {

      if (recordOwner == userID) {
        hasPermission = true;
      } else {
        const captainQuery = 'SELECT StaffUserID FROM UserAuth WHERE CaptainUserID = ?';
        const [staffResult] = await promisePool.execute(captainQuery, [userID]);
        
        if (staffResult.length > 0) {
          const staffUserIDs = staffResult.map(row => row.StaffUserID);
          
          if (staffUserIDs.includes(parseInt(recordOwner))) {
            hasPermission = true;
          }
        }
      }
    }
    
    if (!hasPermission) {
      return res.status(403).json({ error: 'Bu kaydı güncelleme yetkiniz yok' });
    }

    const query = 'UPDATE SquMaintenanceScreen SET SquName = ?, creatorUser = ?, ShopName = ? WHERE id = ?';
    const [result] = await promisePool.execute(query, [SquName, creatorUser, shopName, id]);
    
    res.json({ message: 'Kayıt başarıyla güncellendi' });
  } catch (err) {
    console.error('Kayıt güncellenemedi:', err);
    return res.status(500).json({ error: 'Kayıt güncellenemedi' });
  }
});

// Kayıt sil
router.delete('/squRecords/:id', async (req, res) => {
  const { id } = req.params;
  const { userID, isAdmin } = req.body;
  
  try {
    // Önce kaydın sahibini kontrol et
    const checkQuery = 'SELECT creatorUser FROM SquMaintenanceScreen WHERE id = ?';
    const [checkResult] = await promisePool.execute(checkQuery, [id]);
    
    if (checkResult.length === 0) {
      return res.status(404).json({ error: 'Kayıt bulunamadı' });
    }

    const recordOwner = checkResult[0].creatorUser;
    
    // Yetki kontrolü: Admin veya kayıt sahibi
    if (!isAdmin && recordOwner !== userID) {
      return res.status(403).json({ error: 'Bu kaydı silme yetkiniz yok' });
    }

    const query = 'DELETE FROM SquMaintenanceScreen WHERE id = ?';
    const [result] = await promisePool.execute(query, [id]);
    
    res.json({ message: 'Kayıt başarıyla silindi' });
  } catch (err) {
    console.error('Kayıt silinemedi:', err);
    return res.status(500).json({ error: 'Kayıt silinemedi' });
  }
});

module.exports = router;