const express = require("express");
const fs = require("fs");
const multer = require("multer");
const mysql = require("mysql2/promise");
const xlsx = require("xlsx");
const path = require("path");
const router = express.Router();
const app = express();
app.use(express.json());

const { promisePool } = require("./db");
const e = require("express");

const dbConfig = {
  host: "208.109.240.13",
  user: "root",
  password: "M6mf9d3m!", 
  database: "UBERTY", 
};

let statusLive = 1;

router.get("/shopData", async (req, res) => {
  console.log("Shop data request received");
  let connection;
  try {
    // Create a connection to the database
    connection = await mysql.createConnection(dbConfig);

    // Fetch all nodes
    const [rows] = await connection.execute(
      `SELECT * FROM ${statusLive === 0 ? "NodesTest" : "Nodes"}`
    );

    // Build a mapping of nodes by their ID
    const nodesById = {};
    rows.forEach((node) => {
      node.details = [];
      node.subShops = [];
      nodesById[node.id] = node;
    });

    // Establish parent-child relationships
    const rootNodes = [];
    rows.forEach((node) => {
      if (node.parent_id) {
        const parentNode = nodesById[node.parent_id];
        if (node.children_type === "details") {
          parentNode.details.push(node);
        } else if (node.children_type === "subShops") {
          parentNode.subShops.push(node);
        }
        // Add handling for other children types if needed
      } else {
        rootNodes.push(node);
      }
    });

    res.json(rootNodes);
  } catch (error) {
    console.error("Error fetching shop data:", error);
    res.status(500).send("Error fetching shop data");
  } finally {
    if (connection) {
      await connection.end();
    }
  }
});

router.post("/addNode", async (req, res) => {
  console.log("addNode request received");
  let folderName = statusLive === 0 ? "ImagesTest" : "Images";
  const { tableName, rects, path, designName, buttonName } = req.body;
  console.log("addNode POST:", tableName, path, designName, buttonName);
  const pathElements = path.split("/");
 
  const secondLastElement = pathElements[pathElements.length - 2];
  console.log(pathElements);
  let connection;
  try {
    connection = await promisePool.getConnection();
    await connection.beginTransaction();

    let shopID, clusterID, designID, loopID, parentDesignResultID;

    // Shop check and insert
    const sqlShopCheck = `SELECT id, name FROM ${tableName} WHERE children_type = 'details' AND name = ?`;
    const [shopRows] = await connection.execute(sqlShopCheck, [
      pathElements[1],
    ]);

    if (shopRows.length === 0) {
      const shopSqlNode = `
        INSERT INTO ${tableName} (parent_id, name, children_type, background)
        VALUES (?, ?, ?, ?)
      `;
      const [nodeResult] = await connection.execute(shopSqlNode, [
        "1",
        pathElements[1],
        "details",
        "#F8EDE3",
      ]);

      shopID = nodeResult.insertId;
    } else {
      shopID = shopRows[0].id;
    }

    // Cluster check and insert
    try {
      const sqlClusterCheck = `SELECT id, name FROM ${tableName} WHERE name = ? AND parent_id = ?`;
      const [clusterRows] = await connection.execute(sqlClusterCheck, [
        pathElements[2],
        shopID,
      ]);

      if (clusterRows.length === 0) {
        const clusterSqlNode = `
              INSERT INTO ${tableName} (parent_id, name, children_type, background)
              VALUES (?, ?, ?, ?)
          `;
        const [nodeResult] = await connection.execute(clusterSqlNode, [
          shopID,
          pathElements[2],
          "subShops",
          "#DFD3C3",
        ]);
        clusterID = nodeResult.insertId;
      } else {
        clusterID = clusterRows[0].id;
      }
    } catch (error) {
      console.error("Cluster operation error:", {
        error: error.message,
        code: error.code,
        sqlState: error.sqlState,
        path: pathElements[2],
        shopID: shopID,
      });

      // Send structured error response
      return res.status(400).json({
        success: false,
        error: {
          message: "Cluster işlemi sırasında hata oluştu",
          details: error.message,
          code: error.code,
          path: pathElements[2],
        },
      });
    }

    // Middle elements insertion
    if (pathElements.length >= 5) {
      const middleElements = pathElements.slice(3, pathElements.length - 2);
      console.log(middleElements);

      for (let i = 0; i < middleElements.length; i++) {
        const middleElementsCheck = `
    SELECT id, name FROM ${tableName} WHERE name = ? AND parent_id = ?
  `;
        const parentId = loopID ?? clusterID;
        const [middleElementsRows] = await connection.execute(
          middleElementsCheck,
          [middleElements[i], parentId]
        );

        if (middleElementsRows.length === 0) {
          // Yeni düğüm ekle
          const sqlNode = `
      INSERT INTO ${tableName} (parent_id, name, children_type, background)
      VALUES (?, ?, ?, ?)
    `;
          const currentParentId = i === 0 ? clusterID : loopID;
          const [nodeResult] = await connection.execute(sqlNode, [
            currentParentId,
            middleElements[i],
            "subShops",
            "#DFD3C3",
          ]);
          loopID = nodeResult.insertId;
          console.log(" middleElements[i]: " +  middleElements[i]);
        } else {
          // Mevcut düğümü kullan
          loopID = middleElementsRows[0].id;
        }
      }
    }

    // Also check designID before using it:
    if (!designID) designID = clusterID;
    // Design check and insert
    const sqlDesignCheck = `SELECT id, name FROM ${tableName} WHERE name = ? AND parent_id = ?`;
    const parentId = loopID || clusterID;
    if (!parentId) {
      throw new Error("Parent ID is undefined");
    }
    const [designRows] = await connection.execute(sqlDesignCheck, [
      designName,
      parentId,
    ]);
    console.log(secondLastElement + " " + parentId);
    if (designRows.length === 0) {
      const designSqlNode = `
        INSERT INTO ${tableName} (parent_id, name, description, children_type, background, url, maxHeight)
        VALUES (?, ?, ?, ?, ?, ?, ?)
      `;
      const [parentDesignResult] = await connection.execute(designSqlNode, [
        loopID || designID,
        designName,
        designName,
        "subShops",
        "#F8EDE3",
        folderName + "/Shops" + path,
        "300px",
      ]);
      parentDesignResultID = parentDesignResult.insertId;
    } else {
      parentDesignResultID = designRows[0].id;
    }

    let text1Visible = null;
    let text2Visible = null;
    let text3Visible = null;
    let text4Visible = null;
    // Process rectangles data
    const rectData = Array(4)
      .fill(null)
      .map(() => ({
        height: 0,
        width: 0,
        x: 0,
        y: 0,
        textHeight: 0,
        fontFamily: "",
        rotationAngle: 0,
        textCurve: 0,
      }));

    rects.forEach((part, index) => {
      if (index < 4) {
        rectData[index] = {
          height: part.height,
          width: part.width,
          x: part.x,
          y: part.y,
          textHeight: part.textHeight,
          fontFamily: part.fontFamily,
          rotationAngle: part.rotationAngle,
          textCurve: part.textCurve,
        };

        if (part.height > 0 && part.width > 0) {
          switch (index) {
            case 0:
              text1Visible = true;
              break;
            case 1:
              text2Visible = true;
              break;
            case 2:
              text3Visible = true;
              break;
            case 3:
              text4Visible = true;
              break;
          }
        }
      }
    });

    // Button node insertion
    const buttonSqlNode = `
      INSERT INTO ${tableName} (
        parent_id, name, children_type, url, buttonText,
        height1, width1, x1, y1, textHeight1, fontFamily, rotationAngle1, textCurve1,
        height2, width2, x2, y2, textHeight2, fontFamily2, rotationAngle2, textCurve2,
        height3, width3, x3, y3, textHeight3, fontFamily3, rotationAngle3, textCurve3,
        height4, width4, x4, y4, textHeight4, fontFamily4, rotationAngle4, textCurve4,
        text1Visible, text2Visible, text3Visible, text4Visible
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;

    const buttonValues = [
      parentDesignResultID,
      "SubShop1Button",
      "subShops",
      folderName + "/Shops" + path,
      buttonName,
      ...rectData.flatMap((rect) => [
        rect.height,
        rect.width,
        rect.x,
        rect.y,
        rect.textHeight,
        rect.fontFamily,
        rect.rotationAngle,
        rect.textCurve,
      ]),
      text1Visible,
      text2Visible,
      text3Visible,
      text4Visible,
    ];

    const [buttonResult] = await connection.execute(
      buttonSqlNode,
      buttonValues
    );

    await connection.commit();

    res.status(201).json({
      message: "Node ve ilgili detaylar başarıyla eklendi",
      nodeId: buttonResult.insertId,
    });
  } catch (error) {
    if (connection) await connection.rollback();
    console.error("Error inserting node and details:", error);
    res.status(500).json({ error: "Veri ekleme sırasında bir hata oluştu" });
  } finally {
    if (connection) connection.release();
  }
});

router.post("/updateNode", async (req, res) => {
  let folderName = statusLive === 0 ? "ImagesTest" : "Images";
  const { tableName, rects, path, designName, buttonName, childID } = req.body;

  const pathElements = path.split("/");
  const secondLastElement = pathElements[pathElements.length - 2];

  let connection;
  try {
    connection = await promisePool.getConnection();
    await connection.beginTransaction();

    let text1Visible = null;
    let text2Visible = null;
    let text3Visible = null;
    let text4Visible = null;
    // Process rectangles data
    const rectData = Array(4)
      .fill(null)
      .map(() => ({
        height: 0,
        width: 0,
        x: 0,
        y: 0,
        textHeight: 0,
        fontFamily: "",
        rotationAngle: 0,
        textCurve: 0,
      }));
    console.log(rects);
    rects.forEach((part, index) => {
      if (index < 4) {
        rectData[index] = {
          height: part.height,
          width: part.width,
          x: part.x,
          y: part.y,
          textHeight: part.textHeight,
          fontFamily: part.fontFamily,
          rotationAngle: part.rotationAngle,
          textCurve: part.textCurve,
        };

        if (part.height > 0 && part.width > 0) {
          switch (index) {
            case 0:
              text1Visible = true;
              break;
            case 1:
              text2Visible = true;
              break;
            case 2:
              text3Visible = true;
              break;
            case 3:
              text4Visible = true;
              break;
          }
        }
      }
    });
    let buttonSqlUpdate, buttonValues;
    console.log(path);

    if (path !== null && path !== "") {
      buttonSqlUpdate = `
            UPDATE ${tableName}
            SET
                name = ?,
                children_type = ?,
                url = ?,
                buttonText = ?,
                height1 = ?, width1 = ?, x1 = ?, y1 = ?, textHeight1 = ?, fontFamily = ?, rotationAngle1 = ?, textCurve1 = ?,
                height2 = ?, width2 = ?, x2 = ?, y2 = ?, textHeight2 = ?, fontFamily2 = ?, rotationAngle2 = ?, textCurve2 = ?,
                height3 = ?, width3 = ?, x3 = ?, y3 = ?, textHeight3 = ?, fontFamily3 = ?, rotationAngle3 = ?, textCurve3 = ?,
                height4 = ?, width4 = ?, x4 = ?, y4 = ?, textHeight4 = ?, fontFamily4 = ?, rotationAngle4 = ?, textCurve4 = ?,
                text1Visible = ?, text2Visible = ?, text3Visible = ?, text4Visible = ?
       
            WHERE id = ?
        `;

      buttonValues = [
        "SubShop1Button",
        "subShops",
        folderName + "/Shops" + path,
        buttonName,
        ...rectData.flatMap((rect) => [
          rect.height,
          rect.width,
          rect.x,
          rect.y,
          rect.textHeight,
          rect.fontFamily,
          rect.rotationAngle,
          rect.textCurve,
        ]),
        text1Visible,
        text2Visible,
        text3Visible,
        text4Visible,
        childID,
      ];
    } else {
      buttonSqlUpdate = `
            UPDATE ${tableName}
            SET
                name = ?,
                children_type = ?,
                buttonText = ?,
                height1 = ?, width1 = ?, x1 = ?, y1 = ?, textHeight1 = ?, fontFamily = ?, rotationAngle1 = ?, textCurve1 = ?,
                height2 = ?, width2 = ?, x2 = ?, y2 = ?, textHeight2 = ?, fontFamily2 = ?, rotationAngle2 = ?, textCurve2 = ?,
                height3 = ?, width3 = ?, x3 = ?, y3 = ?, textHeight3 = ?, fontFamily3 = ?, rotationAngle3 = ?, textCurve3 = ?,
                height4 = ?, width4 = ?, x4 = ?, y4 = ?, textHeight4 = ?, fontFamily4 = ?, rotationAngle4 = ?, textCurve4 = ?,
                text1Visible = ?, text2Visible = ?, text3Visible = ?, text4Visible = ?
            WHERE id = ?
        `;

      buttonValues = [
        "SubShop1Button",
        "subShops",
        buttonName,
        ...rectData.flatMap((rect) => [
          rect.height,
          rect.width,
          rect.x,
          rect.y,
          rect.textHeight,
          rect.fontFamily,
          rect.rotationAngle,
          rect.textCurve,
        ]),
        text1Visible,
        text2Visible,
        text3Visible,
        text4Visible,
        childID,
      ];
    }

    const [buttonResult] = await connection.execute(
      buttonSqlUpdate,
      buttonValues
    );

    await connection.commit();

    res.status(201).json({
      message: "Node ve ilgili detaylar başarıyla güncellendi",
      nodeId: buttonResult.insertId,
    });
  } catch (error) {
    if (connection) await connection.rollback();
    console.error("Error inserting node and details:", error);
    res.status(500).json({ error: "Veri ekleme sırasında bir hata oluştu" });
  } finally {
    if (connection) connection.release();
  }
});

let idCounter = 0; 
const getFolderStructure = (dirPath, showFiles = 0) => {
  const items = fs.readdirSync(dirPath, { withFileTypes: true });

  return items
    .filter((item) => item.isDirectory() || (showFiles === 1 && item.isFile()))
    .map((item) => {
      const itemPath = path.join(dirPath, item.name);
      const uniqueId = ++idCounter; // Increment the counter for each item

      if (item.isDirectory()) {
        return {
          key: uniqueId,
          label: item.name,
          children: getFolderStructure(itemPath, showFiles),
        };
      } else {
        return {
          key: uniqueId,
          label: item.name,
          icon: "pi pi-file", // Icon for files
        };
      }
    });
};

const getFolderStructureFiltered = (dirPath, showFiles = 0, authorizedShopNames, isSubFolder = false) => {
  try {
    const items = fs.readdirSync(dirPath, { withFileTypes: true });

    return items
      .filter((item) => {
        // Dosya ise ve showFiles açıksa göster
        if (item.isFile()) {
          return showFiles === 1;
        }
        
        // Alt klasördeyse (isSubFolder = true) tüm klasörleri göster
        if (isSubFolder) {
          return true;
        }
        
        // Ana klasördeyse sadece yetkili shop isimlerini göster
        return authorizedShopNames.includes(item.name);
      })
      .map((item) => {
        const itemPath = path.join(dirPath, item.name);
        const uniqueId = ++idCounter;

        if (item.isDirectory()) {
          return {
            key: uniqueId,
            label: item.name,
            // İlk seviyede yetkili ise, altındaki tüm klasörleri göster (isSubFolder = true)
            children: getFolderStructureFiltered(
              itemPath, 
              showFiles, 
              authorizedShopNames, 
              true  // Alt klasörde olduğumuzu işaretle
            ),
          };
        } else {
          return {
            key: uniqueId,
            label: item.name,
            icon: "pi pi-file",
          };
        }
      });
  } catch (error) {
    console.error("Error reading directory:", error);
    return [];
  }
};

let pathChange =
  statusLive === 0
    ? "C:/Users/bagde/Project/leather-image-converter/public/ImagesTest/"
    : "/home/outlettee/public_html/SVG/Images/";

let shortPath =
  statusLive === 0
    ? "C:/Users/bagde/Project/leather-image-converter/public/ImagesTest/Shops"
    : "/home/outlettee/public_html/SVG/Images/Shops";

router.post("/foldersFiltered", async (req, res) => {
  console.log("Filtered folders request received");
  let connection;
  try {
    const { UserID, Admin, DesignAdmin, showFiles = 1 } = req.body;
    
    if (!UserID) {
      return res.status(400).json({ error: "UserID is required" });
    }

    const isAdmin = Admin === 1 || DesignAdmin === 1;

    // Admin ise tüm folderları getir
    if (isAdmin) {
      const rootPath = path.join(pathChange, "Shops");
      const folderTree = getFolderStructure(rootPath, parseInt(showFiles));
      return res.json(folderTree);
    }

    // Normal kullanıcı için yetkili shop isimlerini al
    connection = await mysql.createConnection(dbConfig);
    const tableName = statusLive === 0 ? "NodesTest" : "Nodes";

    const [authorizedShops] = await connection.execute(
      `SELECT DISTINCT n.name, n.buttonName 
       FROM UserShopAuth usa
       INNER JOIN ${tableName} n ON usa.ShopNodeID = n.id
       WHERE usa.UserID = ? AND usa.CanView = 1 AND n.parent_id = 1`,
      [UserID]
    );

    if (authorizedShops.length === 0) {
      return res.json([]);
    }

    // Yetkili shop isimlerini array'e çevir
    const authorizedShopNames = authorizedShops.map(s => s.name);
    console.log("Authorized shop names:", authorizedShopNames);

    // Folder yapısını yetkili shop'lara göre filtrele
    const rootPath = path.join(pathChange, "Shops");
    const folderTree = getFolderStructureFiltered(
      rootPath, 
      parseInt(showFiles), 
      authorizedShopNames
    );

    res.json(folderTree);
  } catch (error) {
    console.error("Error fetching filtered folders:", error);
    res.status(500).json({ error: "Error fetching folders" });
  } finally {
    if (connection) {
      await connection.end();
    }
  }
});
// ==================== MEVCUT MULTER CONFIGURATION'I GÜNCELLE ====================
// ÖNEMLI: Multer'da req.body okuma sorunu çözümü
// Mevcut upload configuration'ınızı BU ŞEKİLDE değiştirin:

const upload = multer({
  storage: multer.diskStorage({
    destination: function (req, file, cb) {
      // ÖNEMLİ: Multer, body-parser'dan ÖNCE çalışır
      // Bu yüzden folderPath'i buradan alamayız
      // Geçici olarak base path'i kullanacağız
      
      // Şimdilik sadece base path'i döndür
      // Gerçek path'i route handler'da halledeceğiz
      cb(null, shortPath);
    },
    filename: function (req, file, cb) {
      // Dosya adını geçici olarak kaydet
      const cleanName = file.originalname
        .replace(/ğ/g, 'g')
        .replace(/ü/g, 'u')
        .replace(/ş/g, 's')
        .replace(/ı/g, 'i')
        .replace(/ö/g, 'o')
        .replace(/ç/g, 'c')
        .replace(/Ğ/g, 'G')
        .replace(/Ü/g, 'U')
        .replace(/Ş/g, 'S')
        .replace(/İ/g, 'I')
        .replace(/Ö/g, 'O')
        .replace(/Ç/g, 'C')
        .replace(/\s+/g, '_');
      
      cb(null, `temp_${Date.now()}_${cleanName}`);
    },
  }),
  fileFilter: (req, file, cb) => {
    const allowedTypes = ['.svg', '.png', '.jpg', '.jpeg', '.gif', '.webp', '.bmp'];
    const ext = path.extname(file.originalname).toLowerCase();
    
    if (allowedTypes.includes(ext)) {
      cb(null, true);
    } else {
      cb(new Error(`Sadece şu dosya türlerine izin verilir: ${allowedTypes.join(', ')}`), false);
    }
  },
  limits: {
    fileSize: 5 * 1024 * 1024, // 5MB limit
  }
});

router.post(
  "/UploadImage",
  upload.single("file"), 
  async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ 
          success: false, 
          message: "Dosya yüklenmedi" 
        });
      }

      // Şimdi req.body okunabilir
      const folderPath = req.body.folderPath || '';
      
      console.log('=== FILE UPLOAD DEBUG ===');
      console.log('req.body:', req.body);
      console.log('folderPath:', folderPath);
      console.log('Temp file location:', req.file.path);
      console.log('========================');

      // Hedef klasörü oluştur
      const targetDir = path.join(shortPath, folderPath);
      if (!fs.existsSync(targetDir)) {
        fs.mkdirSync(targetDir, { recursive: true });
        console.log('Created directory:', targetDir);
      }

      // Dosya adını temizle (temp_ prefix'ini kaldır)
      const originalName = req.file.filename.replace(/^temp_\d+_/, '');
      const targetPath = path.join(targetDir, originalName);

      console.log('Moving file:');
      console.log('  From:', req.file.path);
      console.log('  To:', targetPath);

      // Dosyayı taşı (geçici konumdan hedef konuma)
      fs.renameSync(req.file.path, targetPath);

      console.log('File moved successfully!');
      console.log('========================');

      res.status(200).json({ 
        success: true,
        message: `"${originalName}" başarıyla yüklendi`,
        file: {
          filename: originalName,
          originalname: req.file.originalname,
          size: req.file.size,
          path: folderPath,
          savedTo: targetPath
        }
      });
    } catch (error) {
      console.error('Upload error:', error);
      
      // Hata durumunda geçici dosyayı temizle
      if (req.file && fs.existsSync(req.file.path)) {
        try {
          fs.unlinkSync(req.file.path);
          console.log('Cleaned up temp file');
        } catch (cleanupError) {
          console.error('Error cleaning up temp file:', cleanupError);
        }
      }
      
      res.status(500).json({ 
        success: false,
        message: "Dosya yüklenirken hata oluştu: " + error.message 
      });
    }
  }
);

// ==================== DELETE IMAGE ROUTE ====================
router.delete("/DeleteImage", express.json(), async (req, res) => {
  try {
    const { filePath } = req.body;

    // VALIDATION
    if (!filePath || !filePath.trim()) {
      return res.status(400).json({
        success: false,
        message: "Dosya yolu gereklidir"
      });
    }

    // Tam dosya yolu oluştur
    const fullPath = path.join(shortPath, filePath);

    console.log('=== DELETE FILE DEBUG ===');
    console.log('Received filePath:', filePath);
    console.log('Full path to delete:', fullPath);
    console.log('========================');

    // GÜVENLİK KONTROLÜ
    const normalizedFullPath = path.normalize(fullPath);
    const normalizedShortPath = path.normalize(shortPath);
    
    if (!normalizedFullPath.startsWith(normalizedShortPath)) {
      return res.status(403).json({
        success: false,
        message: "Geçersiz dosya yolu - Güvenlik ihlali"
      });
    }

    // Dosya kontrolü
    if (!fs.existsSync(fullPath)) {
      return res.status(404).json({
        success: false,
        message: "Dosya bulunamadı"
      });
    }

    const stats = fs.statSync(fullPath);
    if (!stats.isFile()) {
      return res.status(400).json({
        success: false,
        message: "Belirtilen yol bir dosya değil"
      });
    }

    // Dosya tipi kontrolü
    const allowedExtensions = ['.svg', '.png', '.jpg', '.jpeg', '.gif', '.webp', '.bmp'];
    const fileExt = path.extname(fullPath).toLowerCase();
    
    if (!allowedExtensions.includes(fileExt)) {
      return res.status(400).json({
        success: false,
        message: "Sadece resim dosyaları silinebilir"
      });
    }

    // Dosyayı sil
    fs.unlinkSync(fullPath);

    const filename = path.basename(filePath);
    
    console.log('File deleted successfully:', filename);

    return res.status(200).json({
      success: true,
      message: `"${filename}" başarıyla silindi`,
      data: {
        deletedPath: filePath,
        filename: filename
      }
    });

  } catch (error) {
    console.error('Delete error:', error);
    
    if (error.code === 'ENOENT') {
      return res.status(404).json({
        success: false,
        message: 'Dosya bulunamadı'
      });
    } else if (error.code === 'EACCES' || error.code === 'EPERM') {
      return res.status(403).json({
        success: false,
        message: 'Dosya silme izni yok'
      });
    }
    
    return res.status(500).json({
      success: false,
      message: "Dosya silinirken hata oluştu: " + error.message
    });
  }
});


const formParser = multer().none();

 
let fontFolderPath =
  statusLive === 0
    ? "C:/Users/bagde/Project/leather-image-server/Fonts"
    : "/home/outlettee/public_html/leather-image-server/Fonts";
router.get("/fontList", async (req, res) => {
  fs.readdir(fontFolderPath, (err, files) => {
    if (err) {
      console.error("Error reading folder:", err);
      return res.status(500).json({ error: "Unable to read folder" });
    }

    const fileList = files.map((file) => ({
      name: file,
    }));

    res.json(fileList); // Frontend'e JSON formatında g��nder
  });
});

const FONTS_DIR = path.join(
  __dirname,
  "Fonts" // => /home/outlettee/public_html/Leather-image-server/Fonts
);

// Klasör yoksa oluştur
if (!fs.existsSync(FONTS_DIR)) {
  fs.mkdirSync(FONTS_DIR, { recursive: true });
}

function safeName(originalName) {
  const ext = path.extname(originalName).toLowerCase();
  const base = path.basename(originalName, ext)
    .replace(/\s+/g, "_")
    .replace(/[^a-zA-Z0-9_\-\.]/g, "");
  return `${base}${ext}`;
}

const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, FONTS_DIR);
  },
  filename: (req, file, cb) => {
    cb(null, safeName(file.originalname));
  },
});

const fileFilter = (req, file, cb) => {
  const allowedExt = [".ttf", ".otf"];
  const ext = path.extname(file.originalname).toLowerCase();
  if (!allowedExt.includes(ext)) {
    return cb(new Error("Sadece .ttf ve .otf dosyaları kabul edilir"));
  }
  cb(null, true);
};

const fontUpload = multer({
  storage,
  fileFilter,
  limits: {
    fileSize: 20 * 1024 * 1024, // max 20MB / font
    files: 50, // toplu yükleme için üst sınır
  },
});

router.post("/UploadFont", (req, res) => {
  
  try {
    const { fileName, fileData, fileSize, fileType } = req.body;
    
    if (!fileName || !fileData) {
      return res.status(400).json({ ok: false, message: "Dosya adı ve verisi gerekli" });
    }

    const base64Data = fileData.replace(/^data:.*,/, '');
    
    const buffer = Buffer.from(base64Data, 'base64');
    
    // Dosya uzantısını kontrol et
    const ext = path.extname(fileName).toLowerCase();
    const allowedExt = [".ttf", ".otf"];
    if (!allowedExt.includes(ext)) {
      return res.status(400).json({ ok: false, message: "Sadece .ttf ve .otf dosyaları kabul edilir" });
    }
    
    // Güvenli dosya adı oluştur
    const safeFileName = safeName(fileName);
    const filePath = path.join(FONTS_DIR, safeFileName);
    
    // Dosyayı yaz
    fs.writeFileSync(filePath, buffer);
    
    console.log(`Font saved: ${safeFileName}, Size: ${buffer.length} bytes`);
    
    return res.json({
      ok: true,
      fileName: safeFileName,
      path: `/Fonts/${safeFileName}`,
      size: buffer.length
    });
    
  } catch (err) {
    console.error("UploadFont error:", err);
    return res.status(500).json({ ok: false, message: err.message || "Yükleme hatası" });
  }
});


const isPathSafe = (targetPath) => {
  const normalizedTarget = path.normalize(targetPath);
  const normalizedBase = path.normalize(shortPath);
  return normalizedTarget.startsWith(normalizedBase);
};

/**
 * Klasör varlık kontrolü
 */
const isFolderExists = (folderPath) => {
  return fs.existsSync(folderPath);
};

/**
 * Klasör mü kontrol et
 */
const isDirectory = (folderPath) => {
  try {
    const stats = fs.statSync(folderPath);
    return stats.isDirectory();
  } catch (error) {
    return false;
  }
};

// ==================== 1. CREATE FOLDER ====================
router.post('/CreateFolder', formParser, async (req, res) => {
  try {
    let folderPath = req.body.folderPath;
    const folderName = req.body.folderName;

    // VALIDATION - Klasör adı kontrolü
    if (!folderName || !folderName.trim()) {
      return res.status(400).json({
        success: false,
        message: 'Klasör adı gereklidir',
      });
    }

    // Geçersiz karakterleri kontrol et
    const invalidChars = /[<>:"|?*]/;
    if (invalidChars.test(folderName)) {
      return res.status(400).json({
        success: false,
        message: 'Klasör adı geçersiz karakterler içeriyor',
      });
    }

    // FolderPath boş veya undefined ise ana dizin kullan
    if (!folderPath || folderPath === 'undefined' || folderPath.trim() === '') {
      folderPath = '';
    }

    // Tam path oluştur
    const fullPath = path.join(shortPath, folderPath, folderName);
    
    // GÜVENLİK KONTROLÜ - Path traversal saldırılarını önle
    if (!isPathSafe(fullPath)) {
      return res.status(403).json({
        success: false,
        message: 'Geçersiz klasör yolu - Güvenlik ihlali tespit edildi',
      });
    }

    // Klasör zaten var mı kontrol et
    if (isFolderExists(fullPath)) {
      return res.status(400).json({
        success: false,
        message: 'Bu isimde bir klasör zaten mevcut',
      });
    }

    // Parent klasörün var olup olmadığını kontrol et
    const parentPath = path.join(shortPath, folderPath);
    if (!isFolderExists(parentPath)) {
      return res.status(404).json({
        success: false,
        message: 'Üst klasör bulunamadı',
      });
    }

    // Klasörü oluştur
    fs.mkdirSync(fullPath, { recursive: true });

    // Başarılı yanıt
    return res.status(201).json({
      success: true,
      message: 'Klasör başarıyla oluşturuldu',
      data: {
        folderName: folderName,
        folderPath: path.relative(shortPath, fullPath),
        fullPath: fullPath,
      },
    });

  } catch (error) {
    console.error('Error creating folder:', error);

    // Detaylı hata mesajları
    if (error.code === 'EACCES' || error.code === 'EPERM') {
      return res.status(403).json({
        success: false,
        message: 'Klasör oluşturma izni yok',
      });
    } else if (error.code === 'ENOSPC') {
      return res.status(507).json({
        success: false,
        message: 'Disk alanı yetersiz',
      });
    }

    return res.status(500).json({
      success: false,
      message: 'Klasör oluşturulurken bir hata oluştu',
      error: error.message,
    });
  }
});

// ==================== 2. DELETE FOLDER ====================
router.delete('/DeleteFolder', express.json(), async (req, res) => {
  try {
    const { folderPath } = req.body;

    // VALIDATION - Klasör yolu kontrolü
    if (!folderPath || !folderPath.trim()) {
      return res.status(400).json({
        success: false,
        message: 'Klasör yolu gereklidir',
      });
    }

    // Tam path oluştur
    const fullPath = path.join(shortPath, folderPath);

    // GÜVENLİK KONTROLÜ - Path traversal saldırılarını önle
    const normalizedFullPath = path.normalize(fullPath);
    const normalizedShortPath = path.normalize(shortPath);
    
    if (!normalizedFullPath.startsWith(normalizedShortPath)) {
      return res.status(403).json({
        success: false,
        message: 'Geçersiz klasör yolu - Güvenlik ihlali tespit edildi',
      });
    }

    if (normalizedFullPath === normalizedShortPath) {
      return res.status(403).json({
        success: false,
        message: 'Ana dizin silinemez',
      });
    }

    if (!isFolderExists(fullPath)) {
      return res.status(404).json({
        success: false,
        message: 'Klasör bulunamadı',
      });
    }

    if (!isDirectory(fullPath)) {
      return res.status(400).json({
        success: false,
        message: 'Belirtilen yol bir klasör değil',
      });
    }

    
    fs.rmSync(fullPath, { recursive: true, force: true });

    // Başarılı yanıt
    return res.status(200).json({
      success: true,
      message: 'Klasör başarıyla silindi',
      data: {
        deletedPath: folderPath,
      },
    });

  } catch (error) {
    console.error('Error deleting folder:', error);
    
    if (error.code === 'ENOENT') {
      return res.status(404).json({
        success: false,
        message: 'Klasör bulunamadı',
      });
    } else if (error.code === 'EACCES' || error.code === 'EPERM') {
      return res.status(403).json({
        success: false,
        message: 'Klasör silme izni yok',
      });
    } else if (error.code === 'ENOTEMPTY') {
      return res.status(400).json({
        success: false,
        message: 'Klasör boş değil',
      });
    }
    
    return res.status(500).json({
      success: false,
      message: 'Klasör silinirken bir hata oluştu',
      error: error.message,
    });
  }
});

// ==================== 3. RENAME FOLDER (UPDATE) ====================
router.put('/RenameFolder', express.json(), async (req, res) => {
  try {
    const { oldPath, newName } = req.body;

    // VALIDATION - Eski yol kontrolü
    if (!oldPath || !oldPath.trim()) {
      return res.status(400).json({
        success: false,
        message: 'Eski klasör yolu gereklidir',
      });
    }

    // VALIDATION - Yeni isim kontrolü
    if (!newName || !newName.trim()) {
      return res.status(400).json({
        success: false,
        message: 'Yeni klasör adı gereklidir',
      });
    }

    // Geçersiz karakterleri kontrol et
    const invalidChars = /[<>:"|?*\/\\]/;
    if (invalidChars.test(newName)) {
      return res.status(400).json({
        success: false,
        message: 'Yeni klasör adı geçersiz karakterler içeriyor',
      });
    }

    // Tam pathler oluştur
    const oldFullPath = path.join(shortPath, oldPath);
    
    // Parent dizini bul ve yeni path oluştur
    const parentDir = path.dirname(oldFullPath);
    const newFullPath = path.join(parentDir, newName);

    // GÜVENLİK KONTROLÜ - Path traversal saldırılarını önle
    if (!isPathSafe(oldFullPath) || !isPathSafe(newFullPath)) {
      return res.status(403).json({
        success: false,
        message: 'Geçersiz klasör yolu - Güvenlik ihlali tespit edildi',
      });
    }

    // Ana dizinin yeniden adlandırılmasını engelle
    const normalizedOldPath = path.normalize(oldFullPath);
    const normalizedShortPath = path.normalize(shortPath);
    
    if (normalizedOldPath === normalizedShortPath) {
      return res.status(403).json({
        success: false,
        message: 'Ana dizin yeniden adlandırılamaz',
      });
    }

    // Eski klasörün var olup olmadığını kontrol et
    if (!isFolderExists(oldFullPath)) {
      return res.status(404).json({
        success: false,
        message: 'Klasör bulunamadı',
      });
    }

    // Gerçekten bir klasör olup olmadığını kontrol et
    if (!isDirectory(oldFullPath)) {
      return res.status(400).json({
        success: false,
        message: 'Belirtilen yol bir klasör değil',
      });
    }

    // Yeni isimde bir klasör zaten var mı kontrol et
    if (isFolderExists(newFullPath)) {
      return res.status(400).json({
        success: false,
        message: 'Bu isimde bir klasör zaten mevcut',
      });
    }

    // Klasörü yeniden adlandır
    fs.renameSync(oldFullPath, newFullPath);

    // Başarılı yanıt
    return res.status(200).json({
      success: true,
      message: 'Klasör başarıyla yeniden adlandırıldı',
      data: {
        oldPath: oldPath,
        newPath: path.relative(shortPath, newFullPath),
        newName: newName,
      },
    });

  } catch (error) {
    console.error('Error renaming folder:', error);

    // Detaylı hata mesajları
    if (error.code === 'ENOENT') {
      return res.status(404).json({
        success: false,
        message: 'Klasör bulunamadı',
      });
    } else if (error.code === 'EACCES' || error.code === 'EPERM') {
      return res.status(403).json({
        success: false,
        message: 'Klasör yeniden adlandırma izni yok',
      });
    } else if (error.code === 'EEXIST') {
      return res.status(400).json({
        success: false,
        message: 'Hedef klasör zaten mevcut',
      });
    } else if (error.code === 'ENOTEMPTY') {
      return res.status(400).json({
        success: false,
        message: 'Hedef klasör boş değil',
      });
    }
    
    return res.status(500).json({
      success: false,
      message: 'Klasör yeniden adlandırılırken bir hata oluştu',
      error: error.message,
    });
  }
});

// ==================== 4. GET FOLDER INFO (BONUS) ====================
router.get('/GetFolderInfo', async (req, res) => {
  try {
    const folderPath = req.query.path || '';
    const fullPath = path.join(shortPath, folderPath);

    // GÜVENLİK KONTROLÜ
    if (!isPathSafe(fullPath)) {
      return res.status(403).json({
        success: false,
        message: 'Geçersiz klasör yolu',
      });
    }

    // Klasörün var olup olmadığını kontrol et
    if (!isFolderExists(fullPath)) {
      return res.status(404).json({
        success: false,
        message: 'Klasör bulunamadı',
      });
    }

    // Gerçekten bir klasör mü kontrol et
    if (!isDirectory(fullPath)) {
      return res.status(400).json({
        success: false,
        message: 'Belirtilen yol bir klasör değil',
      });
    }

    // Klasör bilgilerini al
    const stats = fs.statSync(fullPath);
    
    // İçeriği oku
    const contents = fs.readdirSync(fullPath);
    const folders = [];
    const files = [];

    contents.forEach(item => {
      const itemPath = path.join(fullPath, item);
      try {
        const itemStats = fs.statSync(itemPath);
        
        const itemInfo = {
          name: item,
          path: path.relative(shortPath, itemPath),
          size: itemStats.size,
          created: itemStats.birthtime,
          modified: itemStats.mtime,
        };

        if (itemStats.isDirectory()) {
          folders.push(itemInfo);
        } else {
          files.push(itemInfo);
        }
      } catch (err) {
        console.error(`Error reading item ${item}:`, err);
      }
    });

    // Başarılı yanıt
    return res.status(200).json({
      success: true,
      data: {
        path: folderPath,
        name: path.basename(fullPath),
        created: stats.birthtime,
        modified: stats.mtime,
        size: stats.size,
        folders: folders,
        files: files,
        totalFolders: folders.length,
        totalFiles: files.length,
      },
    });

  } catch (error) {
    console.error('Error getting folder info:', error);
    
    return res.status(500).json({
      success: false,
      message: 'Klasör bilgileri alınırken bir hata oluştu',
      error: error.message,
    });
  }
});


// Backend API Routes (router.js'e eklenecek)

// Tüm kullanıcıları getir
router.get("/users", async (req, res) => {
  try {
    const connection = await mysql.createConnection(dbConfig);
    const [users] = await connection.execute(
      "SELECT UserID, Username FROM Users ORDER BY Username"
    );
    await connection.end();
    res.json(users);
  } catch (error) {
    console.error("Error fetching users:", error);
    res.status(500).json({ error: "Error fetching users" });
  }
});

// Tüm shop'ları getir (sadece parent_id = 1 olanlar - ana shoplar)
router.get("/shops", async (req, res) => {
  try {
    const connection = await mysql.createConnection(dbConfig);
    const tableName = statusLive === 0 ? "NodesTest" : "Nodes";
    
    // Ana shopları getir (parent_id = 1 olanlar ve children_type != 'subShops')
    const [shops] = await connection.execute(
      `SELECT id, name, buttonName, parent_id 
       FROM ${tableName} 
       WHERE parent_id = 1 
       ORDER BY name`
    );
    await connection.end();
    res.json(shops);
  } catch (error) {
    console.error("Error fetching shops:", error);
    res.status(500).json({ error: "Error fetching shops" });
  }
});

// Kullanıcının shop yetkilerini getir
router.get("/userShopAuth/:userId", async (req, res) => {
  try {
    const { userId } = req.params;
    const connection = await mysql.createConnection(dbConfig);
    
    const [permissions] = await connection.execute(
      `SELECT ShopNodeID, CanView, CanEdit 
       FROM UserShopAuth 
       WHERE UserID = ?`,
      [userId]
    );
    
    await connection.end();
    res.json(permissions);
  } catch (error) {
    console.error("Error fetching user shop permissions:", error);
    res.status(500).json({ error: "Error fetching permissions" });
  }
});

// Shop yetkisi ekle veya güncelle
router.post("/userShopAuth", async (req, res) => {
  try {
    const { UserID, ShopNodeID, CanView, CanEdit } = req.body;
    const connection = await mysql.createConnection(dbConfig);
    
    // Insert or update
    const [result] = await connection.execute(
      `INSERT INTO UserShopAuth (UserID, ShopNodeID, CanView, CanEdit) 
       VALUES (?, ?, ?, ?) 
       ON DUPLICATE KEY UPDATE 
       CanView = VALUES(CanView), 
       CanEdit = VALUES(CanEdit)`,
      [UserID, ShopNodeID, CanView ? 1 : 0, CanEdit ? 1 : 0]
    );
    
    await connection.end();
    res.json({ success: true, message: "Permission saved successfully" });
  } catch (error) {
    console.error("Error saving permission:", error);
    res.status(500).json({ error: "Error saving permission" });
  }
});

// Shop yetkisini sil
router.delete("/userShopAuth/:userId/:shopId", async (req, res) => {
  try {
    const { userId, shopId } = req.params;
    const connection = await mysql.createConnection(dbConfig);
    
    await connection.execute(
      "DELETE FROM UserShopAuth WHERE UserID = ? AND ShopNodeID = ?",
      [userId, shopId]
    );
    
    await connection.end();
    res.json({ success: true, message: "Permission deleted successfully" });
  } catch (error) {
    console.error("Error deleting permission:", error);
    res.status(500).json({ error: "Error deleting permission" });
  }
});

// Kullanıcının tüm yetkilerini toplu kaydet - DÜZELTİLMİŞ VERSİYON
router.post("/userShopAuth/bulk", async (req, res) => {
  let connection;
  try {
    const { UserID, permissions } = req.body;
    
    // Debug için log
    console.log('Saving permissions for UserID:', UserID);
    console.log('Permissions:', permissions);
    
    connection = await mysql.createConnection(dbConfig);
    
    await connection.beginTransaction();
    
    try {
      // Önce mevcut yetkileri sil
      await connection.execute(
        "DELETE FROM UserShopAuth WHERE UserID = ?",
        [UserID]
      );
      
      // Yeni yetkileri ekle
      if (permissions && permissions.length > 0) {
        // Foreign key kontrolü için önce Nodes tablosunda var mı kontrol et
        const tableName = statusLive === 0 ? "NodesTest" : "Nodes";
        
        for (const perm of permissions) {
          // Her ShopNodeID'nin Nodes tablosunda olup olmadığını kontrol et
          const [nodeCheck] = await connection.execute(
            `SELECT id FROM ${tableName} WHERE id = ?`,
            [perm.ShopNodeID]
          );
          
          if (nodeCheck.length === 0) {
            console.error(`ShopNodeID ${perm.ShopNodeID} not found in ${tableName} table!`);
            throw new Error(`Shop ID ${perm.ShopNodeID} bulunamadı!`);
          }
          
          // Node varsa yetkiyi ekle
          await connection.execute(
            `INSERT INTO UserShopAuth (UserID, ShopNodeID, CanView, CanEdit) 
             VALUES (?, ?, ?, ?)`,
            [UserID, perm.ShopNodeID, perm.CanView ? 1 : 0, perm.CanEdit ? 1 : 0]
          );
        }
      }
      
      await connection.commit();
      res.json({ success: true, message: "Permissions saved successfully" });
    } catch (error) {
      await connection.rollback();
      throw error;
    }
  } catch (error) {
    console.error("Error saving bulk permissions:", error);
    res.status(500).json({ 
      error: "Error saving permissions",
      message: error.message 
    });
  } finally {
    if (connection) {
      await connection.end();
    }
  }
});

// Kullanıcının erişebileceği shopları kontrol et (middleware için)
router.post("/checkShopAccess", async (req, res) => {
  try {
    const { UserID, ShopNodeID } = req.body;
    const connection = await mysql.createConnection(dbConfig);
    
    const [result] = await connection.execute(
      `SELECT CanView, CanEdit 
       FROM UserShopAuth 
       WHERE UserID = ? AND ShopNodeID = ?`,
      [UserID, ShopNodeID]
    );
    
    await connection.end();
    
    if (result.length > 0) {
      res.json({ 
        hasAccess: result[0].CanView === 1, 
        canEdit: result[0].CanEdit === 1 
      });
    } else {
      res.json({ hasAccess: false, canEdit: false });
    }
  } catch (error) {
    console.error("Error checking shop access:", error);
    res.status(500).json({ error: "Error checking access" });
  }
});

router.post("/shopDataFiltered", async (req, res) => {
  console.log("Filtered shop data request received");
  let connection;
  try {
    const { UserID, Admin, DesignAdmin } = req.body;
    
    if (!UserID) {
      return res.status(400).json({ error: "UserID is required" });
    }

    connection = await mysql.createConnection(dbConfig);
    const tableName = statusLive === 0 ? "NodesTest" : "Nodes";

    // Admin veya DesignAdmin ise tüm shopları getir
    const isAdmin = Admin === 1 || DesignAdmin === 1;

    let rows;
    if (isAdmin) {
      // Admin ise tüm nodes'ları getir
      [rows] = await connection.execute(
        `SELECT * FROM ${tableName}`
      );
    } else {
      // Normal kullanıcı ise sadece yetkisi olan shopları getir
      // Yetkili shop ID'lerini al
      const [authorizedShops] = await connection.execute(
        `SELECT ShopNodeID FROM UserShopAuth 
         WHERE UserID = ? AND CanView = 1`,
        [UserID]
      );

      if (authorizedShops.length === 0) {
        // Hiç yetkisi yoksa boş array dön
        return res.json([]);
      }

      const shopIds = authorizedShops.map(s => s.ShopNodeID);
      console.log("Fetching data for shop IDs:", shopIds);
      
      // İKİ AŞAMALI ÇÖZÜM - GARANTİLİ
      
      // 1. Adım: Root + Yetkili Shoplar
      const [rootAndShops] = await connection.execute(
        `SELECT * FROM ${tableName}
         WHERE id = 1 OR id IN (${shopIds.join(',')})`
      );
      
      // 2. Adım: Yetkili shopların ALT node'larını çek (recursive)
      const [childNodes] = await connection.execute(
        `WITH RECURSIVE ChildTree AS (
          SELECT * FROM ${tableName}
          WHERE parent_id IN (${shopIds.join(',')})
          
          UNION ALL
          
          SELECT n.* 
          FROM ${tableName} n
          INNER JOIN ChildTree ct ON n.parent_id = ct.id
        )
        SELECT * FROM ChildTree`
      );
      
      // Hepsini birleştir
      rows = [...rootAndShops, ...childNodes];
      
      console.log("Root + Shops:", rootAndShops.length);
      console.log("Child nodes:", childNodes.length);
      console.log("Total nodes fetched for user:", rows.length);
    }

    // Build a mapping of nodes by their ID
    const nodesById = {};
    rows.forEach((node) => {
      node.details = [];
      node.subShops = [];
      nodesById[node.id] = node;
    });

    // Establish parent-child relationships
    const rootNodes = [];
    rows.forEach((node) => {
      if (node.parent_id) {
        const parentNode = nodesById[node.parent_id];
        if (parentNode) {
          if (node.children_type === "details") {
            parentNode.details.push(node);
          } else if (node.children_type === "subShops") {
            parentNode.subShops.push(node);
          }
        }
      } else {
        rootNodes.push(node);
      }
    });

    res.json(rootNodes);
  } catch (error) {
    console.error("Error fetching filtered shop data:", error);
    res.status(500).json({ error: "Error fetching shop data" });
  } finally {
    if (connection) {
      await connection.end();
    }
  }
});
// ESKİ: Tüm shopları getir (admin paneli için)
router.get("/shopData", async (req, res) => {
  console.log("Shop data request received");
  let connection;
  try {
    connection = await mysql.createConnection(dbConfig);
    const tableName = statusLive === 0 ? "NodesTest" : "Nodes";

    const [rows] = await connection.execute(
      `SELECT * FROM ${tableName}`
    );

    const nodesById = {};
    rows.forEach((node) => {
      node.details = [];
      node.subShops = [];
      nodesById[node.id] = node;
    });

    const rootNodes = [];
    rows.forEach((node) => {
      if (node.parent_id) {
        const parentNode = nodesById[node.parent_id];
        if (parentNode) {
          if (node.children_type === "details") {
            parentNode.details.push(node);
          } else if (node.children_type === "subShops") {
            parentNode.subShops.push(node);
          }
        }
      } else {
        rootNodes.push(node);
      }
    });

    res.json(rootNodes);
  } catch (error) {
    console.error("Error fetching shop data:", error);
    res.status(500).send("Error fetching shop data");
  } finally {
    if (connection) {
      await connection.end();
    }
  }
});

module.exports = router;
