"use strict";

var _assert = _interopRequireDefault(require("assert"));

var _src = _interopRequireDefault(require("../src"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } }

function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); return Constructor; }

function assertAlmostEqual(a, b, epsilon) {
  _assert.default.strictEqual(Math.abs(a - b) < epsilon, true);
}

var TextToSVGTest =
/*#__PURE__*/
function () {
  function TextToSVGTest(text) {
    _classCallCheck(this, TextToSVGTest);

    this.textToSVG = _src.default.loadSync();
    this.text = text;
  }

  _createClass(TextToSVGTest, [{
    key: "getMetrics",
    value: function getMetrics(options, expected) {
      var epsilon = 0.001;
      var title = JSON.stringify(options);
      var actual = this.textToSVG.getMetrics(this.text, options);
      it(title, function () {
        assertAlmostEqual(actual.width, expected.width, epsilon);
        assertAlmostEqual(actual.height, expected.height, epsilon);
      });
    }
  }, {
    key: "getD",
    value: function getD(options, expected) {
      var title = JSON.stringify(options);
      var actual = this.textToSVG.getD(this.text, options);
      it(title, function () {
        _assert.default.strictEqual(actual, expected);
      });
    }
  }, {
    key: "getPath",
    value: function getPath(options, expected) {
      var title = JSON.stringify(options);
      var actual = this.textToSVG.getPath(this.text, options);
      it(title, function () {
        _assert.default.strictEqual(actual, expected);
      });
    }
  }, {
    key: "getSVG",
    value: function getSVG(options, expected) {
      var title = JSON.stringify(options);
      var actual = this.textToSVG.getSVG(this.text, options);
      it(title, function () {
        _assert.default.strictEqual(actual, expected);
      });
    }
  }]);

  return TextToSVGTest;
}();

describe('TextToSVG', function () {
  describe('getMetrics', function () {
    describe('hello', function () {
      var hello = new TextToSVGTest('hello');
      hello.getMetrics({}, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 180,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
      hello.getMetrics({
        fontSize: 10
      }, {
        x: 0,
        y: -8.798828125,
        baseline: 0,
        width: 25,
        height: 10,
        ascender: 8.798828125,
        descender: -1.201171875
      });
      hello.getMetrics({
        fontSize: 100
      }, {
        x: 0,
        y: -87.98828125,
        baseline: 0,
        width: 250,
        height: 100,
        ascender: 87.98828125,
        descender: -12.01171875
      });
      hello.getMetrics({
        kerning: false
      }, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 180,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
      hello.getMetrics({
        kerning: true
      }, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 180,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
      hello.getMetrics({
        letterSpacing: 0.2
      }, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 252.00000000000003,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
      hello.getMetrics({
        tracking: 200
      }, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 252.00000000000003,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
    });
    describe('宇治', function () {
      var uji = new TextToSVGTest('宇治');
      uji.getMetrics({}, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 144,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
      uji.getMetrics({
        fontSize: 10
      }, {
        x: 0,
        y: -8.798828125,
        baseline: 0,
        width: 20,
        height: 10,
        ascender: 8.798828125,
        descender: -1.201171875
      });
      uji.getMetrics({
        fontSize: 100
      }, {
        x: 0,
        y: -87.98828125,
        baseline: 0,
        width: 200,
        height: 100,
        ascender: 87.98828125,
        descender: -12.01171875
      });
      uji.getMetrics({
        kerning: false
      }, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 144,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
      uji.getMetrics({
        kerning: true
      }, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 144,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
      uji.getMetrics({
        letterSpacing: 0.2
      }, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 172.8,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
      uji.getMetrics({
        tracking: 200
      }, {
        x: 0,
        y: -63.3515625,
        baseline: 0,
        width: 172.8,
        height: 72,
        ascender: 63.3515625,
        descender: -8.6484375
      });
    });
  });
  describe('getD', function () {
    describe('hello', function () {
      var hello = new TextToSVGTest('hello');
      hello.getD({}, 'M5.27-3.66L5.27-54.07L10.62-54.07L10.62-34.00Q15.86-39.23 21.02-39.23L21.02-39.23Q26.89-39.23 29.60-34.07L29.60-34.07Q31.11-31.15 31.11-27L31.11-27L31.11-3.66L25.77-3.66L25.77-25.42Q25.77-34.14 20.18-34.14L20.18-34.14Q16.42-34.14 13.57-31.39L13.57-31.39Q10.62-28.44 10.62-24.64L10.62-24.64L10.62-3.66L5.27-3.66ZM62.05-14.27L67.68-14.27Q64.55-2.25 54.07-2.25L54.07-2.25Q47.57-2.25 43.77-7.66L43.77-7.66Q40.32-12.62 40.32-20.74L40.32-20.74Q40.32-28.51 43.56-33.47L43.56-33.47Q47.36-39.23 54-39.23L54-39.23Q66.97-39.23 67.82-19.65L67.82-19.65L45.74-19.65Q46.16-7.07 54.14-7.07L54.14-7.07Q60.47-7.07 62.05-14.27L62.05-14.27ZM45.95-24.26L62.05-24.26Q60.89-34.42 54-34.42L54-34.42Q47.36-34.42 45.95-24.26L45.95-24.26ZM92.81-54.07L92.81-11.53Q92.81-8.44 95.77-8.44L95.77-8.44Q98.19-8.44 101.07-9L101.07-9L101.07-3.62Q96.82-3.02 94.82-3.02L94.82-3.02Q87.19-3.02 87.19-10.51L87.19-10.51L87.19-54.07L92.81-54.07ZM128.81-54.07L128.81-11.53Q128.81-8.44 131.77-8.44L131.77-8.44Q134.19-8.44 137.07-9L137.07-9L137.07-3.62Q132.82-3.02 130.82-3.02L130.82-3.02Q123.19-3.02 123.19-10.51L123.19-10.51L123.19-54.07L128.81-54.07ZM162.07-39.23L162.07-39.23Q168.68-39.23 172.44-33.40L172.44-33.40Q175.68-28.55 175.68-20.74L175.68-20.74Q175.68-14.87 173.74-10.44L173.74-10.44Q170.16-2.21 161.93-2.21L161.93-2.21Q155.57-2.21 151.77-7.63L151.77-7.63Q148.32-12.59 148.32-20.74L148.32-20.74Q148.32-29.53 152.30-34.56L152.30-34.56Q156.09-39.23 162.07-39.23ZM161.93-34.21L161.93-34.21Q158.06-34.21 155.88-30.16L155.88-30.16Q153.95-26.61 153.95-20.74L153.95-20.74Q153.95-15.33 155.53-11.92L155.53-11.92Q157.71-7.24 162-7.24L162-7.24Q165.94-7.24 168.12-11.29L168.12-11.29Q170.05-14.84 170.05-20.67L170.05-20.67Q170.05-26.75 168.05-30.23L168.05-30.23Q165.90-34.21 161.93-34.21Z');
      hello.getD({
        x: -30
      }, 'M-24.73-3.66L-24.73-54.07L-19.38-54.07L-19.38-34.00Q-14.14-39.23-8.98-39.23L-8.98-39.23Q-3.11-39.23-0.40-34.07L-0.40-34.07Q1.11-31.15 1.11-27L1.11-27L1.11-3.66L-4.23-3.66L-4.23-25.42Q-4.23-34.14-9.82-34.14L-9.82-34.14Q-13.58-34.14-16.43-31.39L-16.43-31.39Q-19.38-28.44-19.38-24.64L-19.38-24.64L-19.38-3.66L-24.73-3.66ZM32.05-14.27L37.68-14.27Q34.55-2.25 24.07-2.25L24.07-2.25Q17.57-2.25 13.77-7.66L13.77-7.66Q10.32-12.62 10.32-20.74L10.32-20.74Q10.32-28.51 13.56-33.47L13.56-33.47Q17.36-39.23 24-39.23L24-39.23Q36.97-39.23 37.82-19.65L37.82-19.65L15.74-19.65Q16.16-7.07 24.14-7.07L24.14-7.07Q30.47-7.07 32.05-14.27L32.05-14.27ZM15.95-24.26L32.05-24.26Q30.89-34.42 24-34.42L24-34.42Q17.36-34.42 15.95-24.26L15.95-24.26ZM62.81-54.07L62.81-11.53Q62.81-8.44 65.77-8.44L65.77-8.44Q68.19-8.44 71.07-9L71.07-9L71.07-3.62Q66.82-3.02 64.82-3.02L64.82-3.02Q57.19-3.02 57.19-10.51L57.19-10.51L57.19-54.07L62.81-54.07ZM98.81-54.07L98.81-11.53Q98.81-8.44 101.77-8.44L101.77-8.44Q104.19-8.44 107.07-9L107.07-9L107.07-3.62Q102.82-3.02 100.82-3.02L100.82-3.02Q93.19-3.02 93.19-10.51L93.19-10.51L93.19-54.07L98.81-54.07ZM132.07-39.23L132.07-39.23Q138.68-39.23 142.44-33.40L142.44-33.40Q145.68-28.55 145.68-20.74L145.68-20.74Q145.68-14.87 143.74-10.44L143.74-10.44Q140.16-2.21 131.93-2.21L131.93-2.21Q125.57-2.21 121.77-7.63L121.77-7.63Q118.32-12.59 118.32-20.74L118.32-20.74Q118.32-29.53 122.30-34.56L122.30-34.56Q126.09-39.23 132.07-39.23ZM131.93-34.21L131.93-34.21Q128.06-34.21 125.88-30.16L125.88-30.16Q123.95-26.61 123.95-20.74L123.95-20.74Q123.95-15.33 125.53-11.92L125.53-11.92Q127.71-7.24 132-7.24L132-7.24Q135.94-7.24 138.12-11.29L138.12-11.29Q140.05-14.84 140.05-20.67L140.05-20.67Q140.05-26.75 138.05-30.23L138.05-30.23Q135.90-34.21 131.93-34.21Z');
      hello.getD({
        x: 0
      }, 'M5.27-3.66L5.27-54.07L10.62-54.07L10.62-34.00Q15.86-39.23 21.02-39.23L21.02-39.23Q26.89-39.23 29.60-34.07L29.60-34.07Q31.11-31.15 31.11-27L31.11-27L31.11-3.66L25.77-3.66L25.77-25.42Q25.77-34.14 20.18-34.14L20.18-34.14Q16.42-34.14 13.57-31.39L13.57-31.39Q10.62-28.44 10.62-24.64L10.62-24.64L10.62-3.66L5.27-3.66ZM62.05-14.27L67.68-14.27Q64.55-2.25 54.07-2.25L54.07-2.25Q47.57-2.25 43.77-7.66L43.77-7.66Q40.32-12.62 40.32-20.74L40.32-20.74Q40.32-28.51 43.56-33.47L43.56-33.47Q47.36-39.23 54-39.23L54-39.23Q66.97-39.23 67.82-19.65L67.82-19.65L45.74-19.65Q46.16-7.07 54.14-7.07L54.14-7.07Q60.47-7.07 62.05-14.27L62.05-14.27ZM45.95-24.26L62.05-24.26Q60.89-34.42 54-34.42L54-34.42Q47.36-34.42 45.95-24.26L45.95-24.26ZM92.81-54.07L92.81-11.53Q92.81-8.44 95.77-8.44L95.77-8.44Q98.19-8.44 101.07-9L101.07-9L101.07-3.62Q96.82-3.02 94.82-3.02L94.82-3.02Q87.19-3.02 87.19-10.51L87.19-10.51L87.19-54.07L92.81-54.07ZM128.81-54.07L128.81-11.53Q128.81-8.44 131.77-8.44L131.77-8.44Q134.19-8.44 137.07-9L137.07-9L137.07-3.62Q132.82-3.02 130.82-3.02L130.82-3.02Q123.19-3.02 123.19-10.51L123.19-10.51L123.19-54.07L128.81-54.07ZM162.07-39.23L162.07-39.23Q168.68-39.23 172.44-33.40L172.44-33.40Q175.68-28.55 175.68-20.74L175.68-20.74Q175.68-14.87 173.74-10.44L173.74-10.44Q170.16-2.21 161.93-2.21L161.93-2.21Q155.57-2.21 151.77-7.63L151.77-7.63Q148.32-12.59 148.32-20.74L148.32-20.74Q148.32-29.53 152.30-34.56L152.30-34.56Q156.09-39.23 162.07-39.23ZM161.93-34.21L161.93-34.21Q158.06-34.21 155.88-30.16L155.88-30.16Q153.95-26.61 153.95-20.74L153.95-20.74Q153.95-15.33 155.53-11.92L155.53-11.92Q157.71-7.24 162-7.24L162-7.24Q165.94-7.24 168.12-11.29L168.12-11.29Q170.05-14.84 170.05-20.67L170.05-20.67Q170.05-26.75 168.05-30.23L168.05-30.23Q165.90-34.21 161.93-34.21Z');
      hello.getD({
        x: 30
      }, 'M35.27-3.66L35.27-54.07L40.62-54.07L40.62-34.00Q45.86-39.23 51.02-39.23L51.02-39.23Q56.89-39.23 59.60-34.07L59.60-34.07Q61.11-31.15 61.11-27L61.11-27L61.11-3.66L55.77-3.66L55.77-25.42Q55.77-34.14 50.18-34.14L50.18-34.14Q46.42-34.14 43.57-31.39L43.57-31.39Q40.62-28.44 40.62-24.64L40.62-24.64L40.62-3.66L35.27-3.66ZM92.05-14.27L97.68-14.27Q94.55-2.25 84.07-2.25L84.07-2.25Q77.57-2.25 73.77-7.66L73.77-7.66Q70.32-12.62 70.32-20.74L70.32-20.74Q70.32-28.51 73.56-33.47L73.56-33.47Q77.36-39.23 84-39.23L84-39.23Q96.97-39.23 97.82-19.65L97.82-19.65L75.74-19.65Q76.16-7.07 84.14-7.07L84.14-7.07Q90.47-7.07 92.05-14.27L92.05-14.27ZM75.95-24.26L92.05-24.26Q90.89-34.42 84-34.42L84-34.42Q77.36-34.42 75.95-24.26L75.95-24.26ZM122.81-54.07L122.81-11.53Q122.81-8.44 125.77-8.44L125.77-8.44Q128.19-8.44 131.07-9L131.07-9L131.07-3.62Q126.82-3.02 124.82-3.02L124.82-3.02Q117.19-3.02 117.19-10.51L117.19-10.51L117.19-54.07L122.81-54.07ZM158.81-54.07L158.81-11.53Q158.81-8.44 161.77-8.44L161.77-8.44Q164.19-8.44 167.07-9L167.07-9L167.07-3.62Q162.82-3.02 160.82-3.02L160.82-3.02Q153.19-3.02 153.19-10.51L153.19-10.51L153.19-54.07L158.81-54.07ZM192.07-39.23L192.07-39.23Q198.68-39.23 202.44-33.40L202.44-33.40Q205.68-28.55 205.68-20.74L205.68-20.74Q205.68-14.87 203.74-10.44L203.74-10.44Q200.16-2.21 191.93-2.21L191.93-2.21Q185.57-2.21 181.77-7.63L181.77-7.63Q178.32-12.59 178.32-20.74L178.32-20.74Q178.32-29.53 182.30-34.56L182.30-34.56Q186.09-39.23 192.07-39.23ZM191.93-34.21L191.93-34.21Q188.06-34.21 185.88-30.16L185.88-30.16Q183.95-26.61 183.95-20.74L183.95-20.74Q183.95-15.33 185.53-11.92L185.53-11.92Q187.71-7.24 192-7.24L192-7.24Q195.94-7.24 198.12-11.29L198.12-11.29Q200.05-14.84 200.05-20.67L200.05-20.67Q200.05-26.75 198.05-30.23L198.05-30.23Q195.90-34.21 191.93-34.21Z');
      hello.getD({
        y: -30
      }, 'M5.27-33.66L5.27-84.07L10.62-84.07L10.62-64.00Q15.86-69.23 21.02-69.23L21.02-69.23Q26.89-69.23 29.60-64.07L29.60-64.07Q31.11-61.15 31.11-57L31.11-57L31.11-33.66L25.77-33.66L25.77-55.42Q25.77-64.14 20.18-64.14L20.18-64.14Q16.42-64.14 13.57-61.39L13.57-61.39Q10.62-58.44 10.62-54.64L10.62-54.64L10.62-33.66L5.27-33.66ZM62.05-44.27L67.68-44.27Q64.55-32.25 54.07-32.25L54.07-32.25Q47.57-32.25 43.77-37.66L43.77-37.66Q40.32-42.62 40.32-50.74L40.32-50.74Q40.32-58.51 43.56-63.47L43.56-63.47Q47.36-69.23 54-69.23L54-69.23Q66.97-69.23 67.82-49.65L67.82-49.65L45.74-49.65Q46.16-37.07 54.14-37.07L54.14-37.07Q60.47-37.07 62.05-44.27L62.05-44.27ZM45.95-54.26L62.05-54.26Q60.89-64.42 54-64.42L54-64.42Q47.36-64.42 45.95-54.26L45.95-54.26ZM92.81-84.07L92.81-41.53Q92.81-38.44 95.77-38.44L95.77-38.44Q98.19-38.44 101.07-39L101.07-39L101.07-33.62Q96.82-33.02 94.82-33.02L94.82-33.02Q87.19-33.02 87.19-40.51L87.19-40.51L87.19-84.07L92.81-84.07ZM128.81-84.07L128.81-41.53Q128.81-38.44 131.77-38.44L131.77-38.44Q134.19-38.44 137.07-39L137.07-39L137.07-33.62Q132.82-33.02 130.82-33.02L130.82-33.02Q123.19-33.02 123.19-40.51L123.19-40.51L123.19-84.07L128.81-84.07ZM162.07-69.23L162.07-69.23Q168.68-69.23 172.44-63.40L172.44-63.40Q175.68-58.55 175.68-50.74L175.68-50.74Q175.68-44.87 173.74-40.44L173.74-40.44Q170.16-32.21 161.93-32.21L161.93-32.21Q155.57-32.21 151.77-37.63L151.77-37.63Q148.32-42.59 148.32-50.74L148.32-50.74Q148.32-59.53 152.30-64.56L152.30-64.56Q156.09-69.23 162.07-69.23ZM161.93-64.21L161.93-64.21Q158.06-64.21 155.88-60.16L155.88-60.16Q153.95-56.61 153.95-50.74L153.95-50.74Q153.95-45.33 155.53-41.92L155.53-41.92Q157.71-37.24 162-37.24L162-37.24Q165.94-37.24 168.12-41.29L168.12-41.29Q170.05-44.84 170.05-50.67L170.05-50.67Q170.05-56.75 168.05-60.23L168.05-60.23Q165.90-64.21 161.93-64.21Z');
      hello.getD({
        y: 0
      }, 'M5.27-3.66L5.27-54.07L10.62-54.07L10.62-34.00Q15.86-39.23 21.02-39.23L21.02-39.23Q26.89-39.23 29.60-34.07L29.60-34.07Q31.11-31.15 31.11-27L31.11-27L31.11-3.66L25.77-3.66L25.77-25.42Q25.77-34.14 20.18-34.14L20.18-34.14Q16.42-34.14 13.57-31.39L13.57-31.39Q10.62-28.44 10.62-24.64L10.62-24.64L10.62-3.66L5.27-3.66ZM62.05-14.27L67.68-14.27Q64.55-2.25 54.07-2.25L54.07-2.25Q47.57-2.25 43.77-7.66L43.77-7.66Q40.32-12.62 40.32-20.74L40.32-20.74Q40.32-28.51 43.56-33.47L43.56-33.47Q47.36-39.23 54-39.23L54-39.23Q66.97-39.23 67.82-19.65L67.82-19.65L45.74-19.65Q46.16-7.07 54.14-7.07L54.14-7.07Q60.47-7.07 62.05-14.27L62.05-14.27ZM45.95-24.26L62.05-24.26Q60.89-34.42 54-34.42L54-34.42Q47.36-34.42 45.95-24.26L45.95-24.26ZM92.81-54.07L92.81-11.53Q92.81-8.44 95.77-8.44L95.77-8.44Q98.19-8.44 101.07-9L101.07-9L101.07-3.62Q96.82-3.02 94.82-3.02L94.82-3.02Q87.19-3.02 87.19-10.51L87.19-10.51L87.19-54.07L92.81-54.07ZM128.81-54.07L128.81-11.53Q128.81-8.44 131.77-8.44L131.77-8.44Q134.19-8.44 137.07-9L137.07-9L137.07-3.62Q132.82-3.02 130.82-3.02L130.82-3.02Q123.19-3.02 123.19-10.51L123.19-10.51L123.19-54.07L128.81-54.07ZM162.07-39.23L162.07-39.23Q168.68-39.23 172.44-33.40L172.44-33.40Q175.68-28.55 175.68-20.74L175.68-20.74Q175.68-14.87 173.74-10.44L173.74-10.44Q170.16-2.21 161.93-2.21L161.93-2.21Q155.57-2.21 151.77-7.63L151.77-7.63Q148.32-12.59 148.32-20.74L148.32-20.74Q148.32-29.53 152.30-34.56L152.30-34.56Q156.09-39.23 162.07-39.23ZM161.93-34.21L161.93-34.21Q158.06-34.21 155.88-30.16L155.88-30.16Q153.95-26.61 153.95-20.74L153.95-20.74Q153.95-15.33 155.53-11.92L155.53-11.92Q157.71-7.24 162-7.24L162-7.24Q165.94-7.24 168.12-11.29L168.12-11.29Q170.05-14.84 170.05-20.67L170.05-20.67Q170.05-26.75 168.05-30.23L168.05-30.23Q165.90-34.21 161.93-34.21Z');
      hello.getD({
        y: 30
      }, 'M5.27 26.34L5.27-24.07L10.62-24.07L10.62-4.00Q15.86-9.23 21.02-9.23L21.02-9.23Q26.89-9.23 29.60-4.07L29.60-4.07Q31.11-1.15 31.11 3L31.11 3L31.11 26.34L25.77 26.34L25.77 4.58Q25.77-4.14 20.18-4.14L20.18-4.14Q16.42-4.14 13.57-1.39L13.57-1.39Q10.62 1.56 10.62 5.36L10.62 5.36L10.62 26.34L5.27 26.34ZM62.05 15.73L67.68 15.73Q64.55 27.75 54.07 27.75L54.07 27.75Q47.57 27.75 43.77 22.34L43.77 22.34Q40.32 17.38 40.32 9.26L40.32 9.26Q40.32 1.49 43.56-3.47L43.56-3.47Q47.36-9.23 54-9.23L54-9.23Q66.97-9.23 67.82 10.35L67.82 10.35L45.74 10.35Q46.16 22.93 54.14 22.93L54.14 22.93Q60.47 22.93 62.05 15.73L62.05 15.73ZM45.95 5.74L62.05 5.74Q60.89-4.42 54-4.42L54-4.42Q47.36-4.42 45.95 5.74L45.95 5.74ZM92.81-24.07L92.81 18.47Q92.81 21.56 95.77 21.56L95.77 21.56Q98.19 21.56 101.07 21L101.07 21L101.07 26.38Q96.82 26.98 94.82 26.98L94.82 26.98Q87.19 26.98 87.19 19.49L87.19 19.49L87.19-24.07L92.81-24.07ZM128.81-24.07L128.81 18.47Q128.81 21.56 131.77 21.56L131.77 21.56Q134.19 21.56 137.07 21L137.07 21L137.07 26.38Q132.82 26.98 130.82 26.98L130.82 26.98Q123.19 26.98 123.19 19.49L123.19 19.49L123.19-24.07L128.81-24.07ZM162.07-9.23L162.07-9.23Q168.68-9.23 172.44-3.40L172.44-3.40Q175.68 1.45 175.68 9.26L175.68 9.26Q175.68 15.13 173.74 19.56L173.74 19.56Q170.16 27.79 161.93 27.79L161.93 27.79Q155.57 27.79 151.77 22.37L151.77 22.37Q148.32 17.41 148.32 9.26L148.32 9.26Q148.32 0.47 152.30-4.56L152.30-4.56Q156.09-9.23 162.07-9.23ZM161.93-4.21L161.93-4.21Q158.06-4.21 155.88-0.16L155.88-0.16Q153.95 3.39 153.95 9.26L153.95 9.26Q153.95 14.67 155.53 18.08L155.53 18.08Q157.71 22.76 162 22.76L162 22.76Q165.94 22.76 168.12 18.71L168.12 18.71Q170.05 15.16 170.05 9.33L170.05 9.33Q170.05 3.25 168.05-0.23L168.05-0.23Q165.90-4.21 161.93-4.21Z');
      hello.getD({
        fontSize: 10
      }, 'M0.73-0.51L0.73-7.51L1.47-7.51L1.47-4.72Q2.20-5.45 2.92-5.45L2.92-5.45Q3.74-5.45 4.11-4.73L4.11-4.73Q4.32-4.33 4.32-3.75L4.32-3.75L4.32-0.51L3.58-0.51L3.58-3.53Q3.58-4.74 2.80-4.74L2.80-4.74Q2.28-4.74 1.88-4.36L1.88-4.36Q1.47-3.95 1.47-3.42L1.47-3.42L1.47-0.51L0.73-0.51ZM8.62-1.98L9.40-1.98Q8.96-0.31 7.51-0.31L7.51-0.31Q6.61-0.31 6.08-1.06L6.08-1.06Q5.60-1.75 5.60-2.88L5.60-2.88Q5.60-3.96 6.05-4.65L6.05-4.65Q6.58-5.45 7.50-5.45L7.50-5.45Q9.30-5.45 9.42-2.73L9.42-2.73L6.35-2.73Q6.41-0.98 7.52-0.98L7.52-0.98Q8.40-0.98 8.62-1.98L8.62-1.98ZM6.38-3.37L8.62-3.37Q8.46-4.78 7.50-4.78L7.50-4.78Q6.58-4.78 6.38-3.37L6.38-3.37ZM12.89-7.51L12.89-1.60Q12.89-1.17 13.30-1.17L13.30-1.17Q13.64-1.17 14.04-1.25L14.04-1.25L14.04-0.50Q13.45-0.42 13.17-0.42L13.17-0.42Q12.11-0.42 12.11-1.46L12.11-1.46L12.11-7.51L12.89-7.51ZM17.89-7.51L17.89-1.60Q17.89-1.17 18.30-1.17L18.30-1.17Q18.64-1.17 19.04-1.25L19.04-1.25L19.04-0.50Q18.45-0.42 18.17-0.42L18.17-0.42Q17.11-0.42 17.11-1.46L17.11-1.46L17.11-7.51L17.89-7.51ZM22.51-5.45L22.51-5.45Q23.43-5.45 23.95-4.64L23.95-4.64Q24.40-3.96 24.40-2.88L24.40-2.88Q24.40-2.07 24.13-1.45L24.13-1.45Q23.63-0.31 22.49-0.31L22.49-0.31Q21.61-0.31 21.08-1.06L21.08-1.06Q20.60-1.75 20.60-2.88L20.60-2.88Q20.60-4.10 21.15-4.80L21.15-4.80Q21.68-5.45 22.51-5.45ZM22.49-4.75L22.49-4.75Q21.95-4.75 21.65-4.19L21.65-4.19Q21.38-3.70 21.38-2.88L21.38-2.88Q21.38-2.13 21.60-1.66L21.60-1.66Q21.90-1.01 22.50-1.01L22.50-1.01Q23.05-1.01 23.35-1.57L23.35-1.57Q23.62-2.06 23.62-2.87L23.62-2.87Q23.62-3.72 23.34-4.20L23.34-4.20Q23.04-4.75 22.49-4.75Z');
      hello.getD({
        fontSize: 30
      }, 'M2.20-1.52L2.20-22.53L4.42-22.53L4.42-14.17Q6.61-16.35 8.76-16.35L8.76-16.35Q11.21-16.35 12.33-14.19L12.33-14.19Q12.96-12.98 12.96-11.25L12.96-11.25L12.96-1.52L10.74-1.52L10.74-10.59Q10.74-14.22 8.41-14.22L8.41-14.22Q6.84-14.22 5.65-13.08L5.65-13.08Q4.42-11.85 4.42-10.27L4.42-10.27L4.42-1.52L2.20-1.52ZM25.85-5.95L28.20-5.95Q26.89-0.94 22.53-0.94L22.53-0.94Q19.82-0.94 18.24-3.19L18.24-3.19Q16.80-5.26 16.80-8.64L16.80-8.64Q16.80-11.88 18.15-13.95L18.15-13.95Q19.73-16.35 22.50-16.35L22.50-16.35Q27.91-16.35 28.26-8.19L28.26-8.19L19.06-8.19Q19.23-2.94 22.56-2.94L22.56-2.94Q25.20-2.94 25.85-5.95L25.85-5.95ZM19.15-10.11L25.85-10.11Q25.37-14.34 22.50-14.34L22.50-14.34Q19.73-14.34 19.15-10.11L19.15-10.11ZM38.67-22.53L38.67-4.80Q38.67-3.52 39.90-3.52L39.90-3.52Q40.91-3.52 42.11-3.75L42.11-3.75L42.11-1.51Q40.34-1.26 39.51-1.26L39.51-1.26Q36.33-1.26 36.33-4.38L36.33-4.38L36.33-22.53L38.67-22.53ZM53.67-22.53L53.67-4.80Q53.67-3.52 54.90-3.52L54.90-3.52Q55.91-3.52 57.11-3.75L57.11-3.75L57.11-1.51Q55.34-1.26 54.51-1.26L54.51-1.26Q51.33-1.26 51.33-4.38L51.33-4.38L51.33-22.53L53.67-22.53ZM67.53-16.35L67.53-16.35Q70.28-16.35 71.85-13.92L71.85-13.92Q73.20-11.89 73.20-8.64L73.20-8.64Q73.20-6.20 72.39-4.35L72.39-4.35Q70.90-0.92 67.47-0.92L67.47-0.92Q64.82-0.92 63.24-3.18L63.24-3.18Q61.80-5.24 61.80-8.64L61.80-8.64Q61.80-12.30 63.46-14.40L63.46-14.40Q65.04-16.35 67.53-16.35ZM67.47-14.25L67.47-14.25Q65.86-14.25 64.95-12.57L64.95-12.57Q64.15-11.09 64.15-8.64L64.15-8.64Q64.15-6.39 64.80-4.97L64.80-4.97Q65.71-3.02 67.50-3.02L67.50-3.02Q69.14-3.02 70.05-4.70L70.05-4.70Q70.85-6.18 70.85-8.61L70.85-8.61Q70.85-11.15 70.02-12.60L70.02-12.60Q69.13-14.25 67.47-14.25Z');
      hello.getD({
        kerning: false
      }, 'M5.27-3.66L5.27-54.07L10.62-54.07L10.62-34.00Q15.86-39.23 21.02-39.23L21.02-39.23Q26.89-39.23 29.60-34.07L29.60-34.07Q31.11-31.15 31.11-27L31.11-27L31.11-3.66L25.77-3.66L25.77-25.42Q25.77-34.14 20.18-34.14L20.18-34.14Q16.42-34.14 13.57-31.39L13.57-31.39Q10.62-28.44 10.62-24.64L10.62-24.64L10.62-3.66L5.27-3.66ZM62.05-14.27L67.68-14.27Q64.55-2.25 54.07-2.25L54.07-2.25Q47.57-2.25 43.77-7.66L43.77-7.66Q40.32-12.62 40.32-20.74L40.32-20.74Q40.32-28.51 43.56-33.47L43.56-33.47Q47.36-39.23 54-39.23L54-39.23Q66.97-39.23 67.82-19.65L67.82-19.65L45.74-19.65Q46.16-7.07 54.14-7.07L54.14-7.07Q60.47-7.07 62.05-14.27L62.05-14.27ZM45.95-24.26L62.05-24.26Q60.89-34.42 54-34.42L54-34.42Q47.36-34.42 45.95-24.26L45.95-24.26ZM92.81-54.07L92.81-11.53Q92.81-8.44 95.77-8.44L95.77-8.44Q98.19-8.44 101.07-9L101.07-9L101.07-3.62Q96.82-3.02 94.82-3.02L94.82-3.02Q87.19-3.02 87.19-10.51L87.19-10.51L87.19-54.07L92.81-54.07ZM128.81-54.07L128.81-11.53Q128.81-8.44 131.77-8.44L131.77-8.44Q134.19-8.44 137.07-9L137.07-9L137.07-3.62Q132.82-3.02 130.82-3.02L130.82-3.02Q123.19-3.02 123.19-10.51L123.19-10.51L123.19-54.07L128.81-54.07ZM162.07-39.23L162.07-39.23Q168.68-39.23 172.44-33.40L172.44-33.40Q175.68-28.55 175.68-20.74L175.68-20.74Q175.68-14.87 173.74-10.44L173.74-10.44Q170.16-2.21 161.93-2.21L161.93-2.21Q155.57-2.21 151.77-7.63L151.77-7.63Q148.32-12.59 148.32-20.74L148.32-20.74Q148.32-29.53 152.30-34.56L152.30-34.56Q156.09-39.23 162.07-39.23ZM161.93-34.21L161.93-34.21Q158.06-34.21 155.88-30.16L155.88-30.16Q153.95-26.61 153.95-20.74L153.95-20.74Q153.95-15.33 155.53-11.92L155.53-11.92Q157.71-7.24 162-7.24L162-7.24Q165.94-7.24 168.12-11.29L168.12-11.29Q170.05-14.84 170.05-20.67L170.05-20.67Q170.05-26.75 168.05-30.23L168.05-30.23Q165.90-34.21 161.93-34.21Z');
      hello.getD({
        kerning: true
      }, 'M5.27-3.66L5.27-54.07L10.62-54.07L10.62-34.00Q15.86-39.23 21.02-39.23L21.02-39.23Q26.89-39.23 29.60-34.07L29.60-34.07Q31.11-31.15 31.11-27L31.11-27L31.11-3.66L25.77-3.66L25.77-25.42Q25.77-34.14 20.18-34.14L20.18-34.14Q16.42-34.14 13.57-31.39L13.57-31.39Q10.62-28.44 10.62-24.64L10.62-24.64L10.62-3.66L5.27-3.66ZM62.05-14.27L67.68-14.27Q64.55-2.25 54.07-2.25L54.07-2.25Q47.57-2.25 43.77-7.66L43.77-7.66Q40.32-12.62 40.32-20.74L40.32-20.74Q40.32-28.51 43.56-33.47L43.56-33.47Q47.36-39.23 54-39.23L54-39.23Q66.97-39.23 67.82-19.65L67.82-19.65L45.74-19.65Q46.16-7.07 54.14-7.07L54.14-7.07Q60.47-7.07 62.05-14.27L62.05-14.27ZM45.95-24.26L62.05-24.26Q60.89-34.42 54-34.42L54-34.42Q47.36-34.42 45.95-24.26L45.95-24.26ZM92.81-54.07L92.81-11.53Q92.81-8.44 95.77-8.44L95.77-8.44Q98.19-8.44 101.07-9L101.07-9L101.07-3.62Q96.82-3.02 94.82-3.02L94.82-3.02Q87.19-3.02 87.19-10.51L87.19-10.51L87.19-54.07L92.81-54.07ZM128.81-54.07L128.81-11.53Q128.81-8.44 131.77-8.44L131.77-8.44Q134.19-8.44 137.07-9L137.07-9L137.07-3.62Q132.82-3.02 130.82-3.02L130.82-3.02Q123.19-3.02 123.19-10.51L123.19-10.51L123.19-54.07L128.81-54.07ZM162.07-39.23L162.07-39.23Q168.68-39.23 172.44-33.40L172.44-33.40Q175.68-28.55 175.68-20.74L175.68-20.74Q175.68-14.87 173.74-10.44L173.74-10.44Q170.16-2.21 161.93-2.21L161.93-2.21Q155.57-2.21 151.77-7.63L151.77-7.63Q148.32-12.59 148.32-20.74L148.32-20.74Q148.32-29.53 152.30-34.56L152.30-34.56Q156.09-39.23 162.07-39.23ZM161.93-34.21L161.93-34.21Q158.06-34.21 155.88-30.16L155.88-30.16Q153.95-26.61 153.95-20.74L153.95-20.74Q153.95-15.33 155.53-11.92L155.53-11.92Q157.71-7.24 162-7.24L162-7.24Q165.94-7.24 168.12-11.29L168.12-11.29Q170.05-14.84 170.05-20.67L170.05-20.67Q170.05-26.75 168.05-30.23L168.05-30.23Q165.90-34.21 161.93-34.21Z');
      hello.getD({
        letterSpacing: 0.2
      }, 'M5.27-3.66L5.27-54.07L10.62-54.07L10.62-34.00Q15.86-39.23 21.02-39.23L21.02-39.23Q26.89-39.23 29.60-34.07L29.60-34.07Q31.11-31.15 31.11-27L31.11-27L31.11-3.66L25.77-3.66L25.77-25.42Q25.77-34.14 20.18-34.14L20.18-34.14Q16.42-34.14 13.57-31.39L13.57-31.39Q10.62-28.44 10.62-24.64L10.62-24.64L10.62-3.66L5.27-3.66ZM76.45-14.27L82.08-14.27Q78.95-2.25 68.47-2.25L68.47-2.25Q61.97-2.25 58.17-7.66L58.17-7.66Q54.72-12.62 54.72-20.74L54.72-20.74Q54.72-28.51 57.96-33.47L57.96-33.47Q61.76-39.23 68.40-39.23L68.40-39.23Q81.37-39.23 82.22-19.65L82.22-19.65L60.14-19.65Q60.56-7.07 68.54-7.07L68.54-7.07Q74.87-7.07 76.45-14.27L76.45-14.27ZM60.35-24.26L76.45-24.26Q75.29-34.42 68.40-34.42L68.40-34.42Q61.76-34.42 60.35-24.26L60.35-24.26ZM121.61-54.07L121.61-11.53Q121.61-8.44 124.57-8.44L124.57-8.44Q126.99-8.44 129.87-9L129.87-9L129.87-3.62Q125.62-3.02 123.62-3.02L123.62-3.02Q115.99-3.02 115.99-10.51L115.99-10.51L115.99-54.07L121.61-54.07ZM172.01-54.07L172.01-11.53Q172.01-8.44 174.97-8.44L174.97-8.44Q177.39-8.44 180.27-9L180.27-9L180.27-3.62Q176.02-3.02 174.02-3.02L174.02-3.02Q166.39-3.02 166.39-10.51L166.39-10.51L166.39-54.07L172.01-54.07ZM219.67-39.23L219.67-39.23Q226.28-39.23 230.04-33.40L230.04-33.40Q233.28-28.55 233.28-20.74L233.28-20.74Q233.28-14.87 231.34-10.44L231.34-10.44Q227.76-2.21 219.53-2.21L219.53-2.21Q213.17-2.21 209.37-7.63L209.37-7.63Q205.92-12.59 205.92-20.74L205.92-20.74Q205.92-29.53 209.90-34.56L209.90-34.56Q213.69-39.23 219.67-39.23ZM219.53-34.21L219.53-34.21Q215.66-34.21 213.48-30.16L213.48-30.16Q211.55-26.61 211.55-20.74L211.55-20.74Q211.55-15.33 213.13-11.92L213.13-11.92Q215.31-7.24 219.60-7.24L219.60-7.24Q223.54-7.24 225.72-11.29L225.72-11.29Q227.65-14.84 227.65-20.67L227.65-20.67Q227.65-26.75 225.65-30.23L225.65-30.23Q223.50-34.21 219.53-34.21Z');
      hello.getD({
        tracking: 200
      }, 'M5.27-3.66L5.27-54.07L10.62-54.07L10.62-34.00Q15.86-39.23 21.02-39.23L21.02-39.23Q26.89-39.23 29.60-34.07L29.60-34.07Q31.11-31.15 31.11-27L31.11-27L31.11-3.66L25.77-3.66L25.77-25.42Q25.77-34.14 20.18-34.14L20.18-34.14Q16.42-34.14 13.57-31.39L13.57-31.39Q10.62-28.44 10.62-24.64L10.62-24.64L10.62-3.66L5.27-3.66ZM76.45-14.27L82.08-14.27Q78.95-2.25 68.47-2.25L68.47-2.25Q61.97-2.25 58.17-7.66L58.17-7.66Q54.72-12.62 54.72-20.74L54.72-20.74Q54.72-28.51 57.96-33.47L57.96-33.47Q61.76-39.23 68.40-39.23L68.40-39.23Q81.37-39.23 82.22-19.65L82.22-19.65L60.14-19.65Q60.56-7.07 68.54-7.07L68.54-7.07Q74.87-7.07 76.45-14.27L76.45-14.27ZM60.35-24.26L76.45-24.26Q75.29-34.42 68.40-34.42L68.40-34.42Q61.76-34.42 60.35-24.26L60.35-24.26ZM121.61-54.07L121.61-11.53Q121.61-8.44 124.57-8.44L124.57-8.44Q126.99-8.44 129.87-9L129.87-9L129.87-3.62Q125.62-3.02 123.62-3.02L123.62-3.02Q115.99-3.02 115.99-10.51L115.99-10.51L115.99-54.07L121.61-54.07ZM172.01-54.07L172.01-11.53Q172.01-8.44 174.97-8.44L174.97-8.44Q177.39-8.44 180.27-9L180.27-9L180.27-3.62Q176.02-3.02 174.02-3.02L174.02-3.02Q166.39-3.02 166.39-10.51L166.39-10.51L166.39-54.07L172.01-54.07ZM219.67-39.23L219.67-39.23Q226.28-39.23 230.04-33.40L230.04-33.40Q233.28-28.55 233.28-20.74L233.28-20.74Q233.28-14.87 231.34-10.44L231.34-10.44Q227.76-2.21 219.53-2.21L219.53-2.21Q213.17-2.21 209.37-7.63L209.37-7.63Q205.92-12.59 205.92-20.74L205.92-20.74Q205.92-29.53 209.90-34.56L209.90-34.56Q213.69-39.23 219.67-39.23ZM219.53-34.21L219.53-34.21Q215.66-34.21 213.48-30.16L213.48-30.16Q211.55-26.61 211.55-20.74L211.55-20.74Q211.55-15.33 213.13-11.92L213.13-11.92Q215.31-7.24 219.60-7.24L219.60-7.24Q223.54-7.24 225.72-11.29L225.72-11.29Q227.65-14.84 227.65-20.67L227.65-20.67Q227.65-26.75 225.65-30.23L225.65-30.23Q223.50-34.21 219.53-34.21Z');
      hello.getD({
        anchor: 'left'
      }, 'M5.27-3.66L5.27-54.07L10.62-54.07L10.62-34.00Q15.86-39.23 21.02-39.23L21.02-39.23Q26.89-39.23 29.60-34.07L29.60-34.07Q31.11-31.15 31.11-27L31.11-27L31.11-3.66L25.77-3.66L25.77-25.42Q25.77-34.14 20.18-34.14L20.18-34.14Q16.42-34.14 13.57-31.39L13.57-31.39Q10.62-28.44 10.62-24.64L10.62-24.64L10.62-3.66L5.27-3.66ZM62.05-14.27L67.68-14.27Q64.55-2.25 54.07-2.25L54.07-2.25Q47.57-2.25 43.77-7.66L43.77-7.66Q40.32-12.62 40.32-20.74L40.32-20.74Q40.32-28.51 43.56-33.47L43.56-33.47Q47.36-39.23 54-39.23L54-39.23Q66.97-39.23 67.82-19.65L67.82-19.65L45.74-19.65Q46.16-7.07 54.14-7.07L54.14-7.07Q60.47-7.07 62.05-14.27L62.05-14.27ZM45.95-24.26L62.05-24.26Q60.89-34.42 54-34.42L54-34.42Q47.36-34.42 45.95-24.26L45.95-24.26ZM92.81-54.07L92.81-11.53Q92.81-8.44 95.77-8.44L95.77-8.44Q98.19-8.44 101.07-9L101.07-9L101.07-3.62Q96.82-3.02 94.82-3.02L94.82-3.02Q87.19-3.02 87.19-10.51L87.19-10.51L87.19-54.07L92.81-54.07ZM128.81-54.07L128.81-11.53Q128.81-8.44 131.77-8.44L131.77-8.44Q134.19-8.44 137.07-9L137.07-9L137.07-3.62Q132.82-3.02 130.82-3.02L130.82-3.02Q123.19-3.02 123.19-10.51L123.19-10.51L123.19-54.07L128.81-54.07ZM162.07-39.23L162.07-39.23Q168.68-39.23 172.44-33.40L172.44-33.40Q175.68-28.55 175.68-20.74L175.68-20.74Q175.68-14.87 173.74-10.44L173.74-10.44Q170.16-2.21 161.93-2.21L161.93-2.21Q155.57-2.21 151.77-7.63L151.77-7.63Q148.32-12.59 148.32-20.74L148.32-20.74Q148.32-29.53 152.30-34.56L152.30-34.56Q156.09-39.23 162.07-39.23ZM161.93-34.21L161.93-34.21Q158.06-34.21 155.88-30.16L155.88-30.16Q153.95-26.61 153.95-20.74L153.95-20.74Q153.95-15.33 155.53-11.92L155.53-11.92Q157.71-7.24 162-7.24L162-7.24Q165.94-7.24 168.12-11.29L168.12-11.29Q170.05-14.84 170.05-20.67L170.05-20.67Q170.05-26.75 168.05-30.23L168.05-30.23Q165.90-34.21 161.93-34.21Z');
      hello.getD({
        anchor: 'center'
      }, 'M-84.73-3.66L-84.73-54.07L-79.38-54.07L-79.38-34.00Q-74.14-39.23-68.98-39.23L-68.98-39.23Q-63.11-39.23-60.40-34.07L-60.40-34.07Q-58.89-31.15-58.89-27L-58.89-27L-58.89-3.66L-64.23-3.66L-64.23-25.42Q-64.23-34.14-69.82-34.14L-69.82-34.14Q-73.58-34.14-76.43-31.39L-76.43-31.39Q-79.38-28.44-79.38-24.64L-79.38-24.64L-79.38-3.66L-84.73-3.66ZM-27.95-14.27L-22.32-14.27Q-25.45-2.25-35.93-2.25L-35.93-2.25Q-42.43-2.25-46.23-7.66L-46.23-7.66Q-49.68-12.62-49.68-20.74L-49.68-20.74Q-49.68-28.51-46.44-33.47L-46.44-33.47Q-42.64-39.23-36-39.23L-36-39.23Q-23.03-39.23-22.18-19.65L-22.18-19.65L-44.26-19.65Q-43.84-7.07-35.86-7.07L-35.86-7.07Q-29.53-7.07-27.95-14.27L-27.95-14.27ZM-44.05-24.26L-27.95-24.26Q-29.11-34.42-36-34.42L-36-34.42Q-42.64-34.42-44.05-24.26L-44.05-24.26ZM2.81-54.07L2.81-11.53Q2.81-8.44 5.77-8.44L5.77-8.44Q8.19-8.44 11.07-9L11.07-9L11.07-3.62Q6.82-3.02 4.82-3.02L4.82-3.02Q-2.81-3.02-2.81-10.51L-2.81-10.51L-2.81-54.07L2.81-54.07ZM38.81-54.07L38.81-11.53Q38.81-8.44 41.77-8.44L41.77-8.44Q44.19-8.44 47.07-9L47.07-9L47.07-3.62Q42.82-3.02 40.82-3.02L40.82-3.02Q33.19-3.02 33.19-10.51L33.19-10.51L33.19-54.07L38.81-54.07ZM72.07-39.23L72.07-39.23Q78.68-39.23 82.44-33.40L82.44-33.40Q85.68-28.55 85.68-20.74L85.68-20.74Q85.68-14.87 83.74-10.44L83.74-10.44Q80.16-2.21 71.93-2.21L71.93-2.21Q65.57-2.21 61.77-7.63L61.77-7.63Q58.32-12.59 58.32-20.74L58.32-20.74Q58.32-29.53 62.30-34.56L62.30-34.56Q66.09-39.23 72.07-39.23ZM71.93-34.21L71.93-34.21Q68.06-34.21 65.88-30.16L65.88-30.16Q63.95-26.61 63.95-20.74L63.95-20.74Q63.95-15.33 65.53-11.92L65.53-11.92Q67.71-7.24 72-7.24L72-7.24Q75.94-7.24 78.12-11.29L78.12-11.29Q80.05-14.84 80.05-20.67L80.05-20.67Q80.05-26.75 78.05-30.23L78.05-30.23Q75.90-34.21 71.93-34.21Z');
      hello.getD({
        anchor: 'right'
      }, 'M-174.73-3.66L-174.73-54.07L-169.38-54.07L-169.38-34.00Q-164.14-39.23-158.98-39.23L-158.98-39.23Q-153.11-39.23-150.40-34.07L-150.40-34.07Q-148.89-31.15-148.89-27L-148.89-27L-148.89-3.66L-154.23-3.66L-154.23-25.42Q-154.23-34.14-159.82-34.14L-159.82-34.14Q-163.58-34.14-166.43-31.39L-166.43-31.39Q-169.38-28.44-169.38-24.64L-169.38-24.64L-169.38-3.66L-174.73-3.66ZM-117.95-14.27L-112.32-14.27Q-115.45-2.25-125.93-2.25L-125.93-2.25Q-132.43-2.25-136.23-7.66L-136.23-7.66Q-139.68-12.62-139.68-20.74L-139.68-20.74Q-139.68-28.51-136.44-33.47L-136.44-33.47Q-132.64-39.23-126-39.23L-126-39.23Q-113.03-39.23-112.18-19.65L-112.18-19.65L-134.26-19.65Q-133.84-7.07-125.86-7.07L-125.86-7.07Q-119.53-7.07-117.95-14.27L-117.95-14.27ZM-134.05-24.26L-117.95-24.26Q-119.11-34.42-126-34.42L-126-34.42Q-132.64-34.42-134.05-24.26L-134.05-24.26ZM-87.19-54.07L-87.19-11.53Q-87.19-8.44-84.23-8.44L-84.23-8.44Q-81.81-8.44-78.93-9L-78.93-9L-78.93-3.62Q-83.18-3.02-85.18-3.02L-85.18-3.02Q-92.81-3.02-92.81-10.51L-92.81-10.51L-92.81-54.07L-87.19-54.07ZM-51.19-54.07L-51.19-11.53Q-51.19-8.44-48.23-8.44L-48.23-8.44Q-45.81-8.44-42.93-9L-42.93-9L-42.93-3.62Q-47.18-3.02-49.18-3.02L-49.18-3.02Q-56.81-3.02-56.81-10.51L-56.81-10.51L-56.81-54.07L-51.19-54.07ZM-17.93-39.23L-17.93-39.23Q-11.32-39.23-7.56-33.40L-7.56-33.40Q-4.32-28.55-4.32-20.74L-4.32-20.74Q-4.32-14.87-6.26-10.44L-6.26-10.44Q-9.84-2.21-18.07-2.21L-18.07-2.21Q-24.43-2.21-28.23-7.63L-28.23-7.63Q-31.68-12.59-31.68-20.74L-31.68-20.74Q-31.68-29.53-27.70-34.56L-27.70-34.56Q-23.91-39.23-17.93-39.23ZM-18.07-34.21L-18.07-34.21Q-21.94-34.21-24.12-30.16L-24.12-30.16Q-26.05-26.61-26.05-20.74L-26.05-20.74Q-26.05-15.33-24.47-11.92L-24.47-11.92Q-22.29-7.24-18-7.24L-18-7.24Q-14.06-7.24-11.88-11.29L-11.88-11.29Q-9.95-14.84-9.95-20.67L-9.95-20.67Q-9.95-26.75-11.95-30.23L-11.95-30.23Q-14.10-34.21-18.07-34.21Z');
      hello.getD({
        anchor: 'top'
      }, 'M5.27 59.70L5.27 9.28L10.62 9.28L10.62 29.36Q15.86 24.12 21.02 24.12L21.02 24.12Q26.89 24.12 29.60 29.29L29.60 29.29Q31.11 32.20 31.11 36.35L31.11 36.35L31.11 59.70L25.77 59.70L25.77 37.93Q25.77 29.21 20.18 29.21L20.18 29.21Q16.42 29.21 13.57 31.96L13.57 31.96Q10.62 34.91 10.62 38.71L10.62 38.71L10.62 59.70L5.27 59.70ZM62.05 49.08L67.68 49.08Q64.55 61.10 54.07 61.10L54.07 61.10Q47.57 61.10 43.77 55.69L43.77 55.69Q40.32 50.73 40.32 42.61L40.32 42.61Q40.32 34.84 43.56 29.88L43.56 29.88Q47.36 24.12 54 24.12L54 24.12Q66.97 24.12 67.82 43.70L67.82 43.70L45.74 43.70Q46.16 56.29 54.14 56.29L54.14 56.29Q60.47 56.29 62.05 49.08L62.05 49.08ZM45.95 39.09L62.05 39.09Q60.89 28.93 54 28.93L54 28.93Q47.36 28.93 45.95 39.09L45.95 39.09ZM92.81 9.28L92.81 51.82Q92.81 54.91 95.77 54.91L95.77 54.91Q98.19 54.91 101.07 54.35L101.07 54.35L101.07 59.73Q96.82 60.33 94.82 60.33L94.82 60.33Q87.19 60.33 87.19 52.84L87.19 52.84L87.19 9.28L92.81 9.28ZM128.81 9.28L128.81 51.82Q128.81 54.91 131.77 54.91L131.77 54.91Q134.19 54.91 137.07 54.35L137.07 54.35L137.07 59.73Q132.82 60.33 130.82 60.33L130.82 60.33Q123.19 60.33 123.19 52.84L123.19 52.84L123.19 9.28L128.81 9.28ZM162.07 24.12L162.07 24.12Q168.68 24.12 172.44 29.95L172.44 29.95Q175.68 34.80 175.68 42.61L175.68 42.61Q175.68 48.48 173.74 52.91L173.74 52.91Q170.16 61.14 161.93 61.14L161.93 61.14Q155.57 61.14 151.77 55.72L151.77 55.72Q148.32 50.77 148.32 42.61L148.32 42.61Q148.32 33.82 152.30 28.79L152.30 28.79Q156.09 24.12 162.07 24.12ZM161.93 29.14L161.93 29.14Q158.06 29.14 155.88 33.19L155.88 33.19Q153.95 36.74 153.95 42.61L153.95 42.61Q153.95 48.02 155.53 51.43L155.53 51.43Q157.71 56.11 162 56.11L162 56.11Q165.94 56.11 168.12 52.07L168.12 52.07Q170.05 48.52 170.05 42.68L170.05 42.68Q170.05 36.60 168.05 33.12L168.05 33.12Q165.90 29.14 161.93 29.14Z');
      hello.getD({
        anchor: 'middle'
      }, 'M5.27 23.70L5.27-26.72L10.62-26.72L10.62-6.64Q15.86-11.88 21.02-11.88L21.02-11.88Q26.89-11.88 29.60-6.71L29.60-6.71Q31.11-3.80 31.11 0.35L31.11 0.35L31.11 23.70L25.77 23.70L25.77 1.93Q25.77-6.79 20.18-6.79L20.18-6.79Q16.42-6.79 13.57-4.04L13.57-4.04Q10.62-1.09 10.62 2.71L10.62 2.71L10.62 23.70L5.27 23.70ZM62.05 13.08L67.68 13.08Q64.55 25.10 54.07 25.10L54.07 25.10Q47.57 25.10 43.77 19.69L43.77 19.69Q40.32 14.73 40.32 6.61L40.32 6.61Q40.32-1.16 43.56-6.12L43.56-6.12Q47.36-11.88 54-11.88L54-11.88Q66.97-11.88 67.82 7.70L67.82 7.70L45.74 7.70Q46.16 20.29 54.14 20.29L54.14 20.29Q60.47 20.29 62.05 13.08L62.05 13.08ZM45.95 3.09L62.05 3.09Q60.89-7.07 54-7.07L54-7.07Q47.36-7.07 45.95 3.09L45.95 3.09ZM92.81-26.72L92.81 15.82Q92.81 18.91 95.77 18.91L95.77 18.91Q98.19 18.91 101.07 18.35L101.07 18.35L101.07 23.73Q96.82 24.33 94.82 24.33L94.82 24.33Q87.19 24.33 87.19 16.84L87.19 16.84L87.19-26.72L92.81-26.72ZM128.81-26.72L128.81 15.82Q128.81 18.91 131.77 18.91L131.77 18.91Q134.19 18.91 137.07 18.35L137.07 18.35L137.07 23.73Q132.82 24.33 130.82 24.33L130.82 24.33Q123.19 24.33 123.19 16.84L123.19 16.84L123.19-26.72L128.81-26.72ZM162.07-11.88L162.07-11.88Q168.68-11.88 172.44-6.05L172.44-6.05Q175.68-1.20 175.68 6.61L175.68 6.61Q175.68 12.48 173.74 16.91L173.74 16.91Q170.16 25.14 161.93 25.14L161.93 25.14Q155.57 25.14 151.77 19.72L151.77 19.72Q148.32 14.77 148.32 6.61L148.32 6.61Q148.32-2.18 152.30-7.21L152.30-7.21Q156.09-11.88 162.07-11.88ZM161.93-6.86L161.93-6.86Q158.06-6.86 155.88-2.81L155.88-2.81Q153.95 0.74 153.95 6.61L153.95 6.61Q153.95 12.02 155.53 15.43L155.53 15.43Q157.71 20.11 162 20.11L162 20.11Q165.94 20.11 168.12 16.07L168.12 16.07Q170.05 12.52 170.05 6.68L170.05 6.68Q170.05 0.60 168.05-2.88L168.05-2.88Q165.90-6.86 161.93-6.86Z');
      hello.getD({
        anchor: 'bottom'
      }, 'M5.27-12.30L5.27-62.72L10.62-62.72L10.62-42.64Q15.86-47.88 21.02-47.88L21.02-47.88Q26.89-47.88 29.60-42.71L29.60-42.71Q31.11-39.80 31.11-35.65L31.11-35.65L31.11-12.30L25.77-12.30L25.77-34.07Q25.77-42.79 20.18-42.79L20.18-42.79Q16.42-42.79 13.57-40.04L13.57-40.04Q10.62-37.09 10.62-33.29L10.62-33.29L10.62-12.30L5.27-12.30ZM62.05-22.92L67.68-22.92Q64.55-10.90 54.07-10.90L54.07-10.90Q47.57-10.90 43.77-16.31L43.77-16.31Q40.32-21.27 40.32-29.39L40.32-29.39Q40.32-37.16 43.56-42.12L43.56-42.12Q47.36-47.88 54-47.88L54-47.88Q66.97-47.88 67.82-28.30L67.82-28.30L45.74-28.30Q46.16-15.71 54.14-15.71L54.14-15.71Q60.47-15.71 62.05-22.92L62.05-22.92ZM45.95-32.91L62.05-32.91Q60.89-43.07 54-43.07L54-43.07Q47.36-43.07 45.95-32.91L45.95-32.91ZM92.81-62.72L92.81-20.18Q92.81-17.09 95.77-17.09L95.77-17.09Q98.19-17.09 101.07-17.65L101.07-17.65L101.07-12.27Q96.82-11.67 94.82-11.67L94.82-11.67Q87.19-11.67 87.19-19.16L87.19-19.16L87.19-62.72L92.81-62.72ZM128.81-62.72L128.81-20.18Q128.81-17.09 131.77-17.09L131.77-17.09Q134.19-17.09 137.07-17.65L137.07-17.65L137.07-12.27Q132.82-11.67 130.82-11.67L130.82-11.67Q123.19-11.67 123.19-19.16L123.19-19.16L123.19-62.72L128.81-62.72ZM162.07-47.88L162.07-47.88Q168.68-47.88 172.44-42.05L172.44-42.05Q175.68-37.20 175.68-29.39L175.68-29.39Q175.68-23.52 173.74-19.09L173.74-19.09Q170.16-10.86 161.93-10.86L161.93-10.86Q155.57-10.86 151.77-16.28L151.77-16.28Q148.32-21.23 148.32-29.39L148.32-29.39Q148.32-38.18 152.30-43.21L152.30-43.21Q156.09-47.88 162.07-47.88ZM161.93-42.86L161.93-42.86Q158.06-42.86 155.88-38.81L155.88-38.81Q153.95-35.26 153.95-29.39L153.95-29.39Q153.95-23.98 155.53-20.57L155.53-20.57Q157.71-15.89 162-15.89L162-15.89Q165.94-15.89 168.12-19.93L168.12-19.93Q170.05-23.48 170.05-29.32L170.05-29.32Q170.05-35.40 168.05-38.88L168.05-38.88Q165.90-42.86 161.93-42.86Z');
      hello.getD({
        anchor: 'left top'
      }, 'M5.27 59.70L5.27 9.28L10.62 9.28L10.62 29.36Q15.86 24.12 21.02 24.12L21.02 24.12Q26.89 24.12 29.60 29.29L29.60 29.29Q31.11 32.20 31.11 36.35L31.11 36.35L31.11 59.70L25.77 59.70L25.77 37.93Q25.77 29.21 20.18 29.21L20.18 29.21Q16.42 29.21 13.57 31.96L13.57 31.96Q10.62 34.91 10.62 38.71L10.62 38.71L10.62 59.70L5.27 59.70ZM62.05 49.08L67.68 49.08Q64.55 61.10 54.07 61.10L54.07 61.10Q47.57 61.10 43.77 55.69L43.77 55.69Q40.32 50.73 40.32 42.61L40.32 42.61Q40.32 34.84 43.56 29.88L43.56 29.88Q47.36 24.12 54 24.12L54 24.12Q66.97 24.12 67.82 43.70L67.82 43.70L45.74 43.70Q46.16 56.29 54.14 56.29L54.14 56.29Q60.47 56.29 62.05 49.08L62.05 49.08ZM45.95 39.09L62.05 39.09Q60.89 28.93 54 28.93L54 28.93Q47.36 28.93 45.95 39.09L45.95 39.09ZM92.81 9.28L92.81 51.82Q92.81 54.91 95.77 54.91L95.77 54.91Q98.19 54.91 101.07 54.35L101.07 54.35L101.07 59.73Q96.82 60.33 94.82 60.33L94.82 60.33Q87.19 60.33 87.19 52.84L87.19 52.84L87.19 9.28L92.81 9.28ZM128.81 9.28L128.81 51.82Q128.81 54.91 131.77 54.91L131.77 54.91Q134.19 54.91 137.07 54.35L137.07 54.35L137.07 59.73Q132.82 60.33 130.82 60.33L130.82 60.33Q123.19 60.33 123.19 52.84L123.19 52.84L123.19 9.28L128.81 9.28ZM162.07 24.12L162.07 24.12Q168.68 24.12 172.44 29.95L172.44 29.95Q175.68 34.80 175.68 42.61L175.68 42.61Q175.68 48.48 173.74 52.91L173.74 52.91Q170.16 61.14 161.93 61.14L161.93 61.14Q155.57 61.14 151.77 55.72L151.77 55.72Q148.32 50.77 148.32 42.61L148.32 42.61Q148.32 33.82 152.30 28.79L152.30 28.79Q156.09 24.12 162.07 24.12ZM161.93 29.14L161.93 29.14Q158.06 29.14 155.88 33.19L155.88 33.19Q153.95 36.74 153.95 42.61L153.95 42.61Q153.95 48.02 155.53 51.43L155.53 51.43Q157.71 56.11 162 56.11L162 56.11Q165.94 56.11 168.12 52.07L168.12 52.07Q170.05 48.52 170.05 42.68L170.05 42.68Q170.05 36.60 168.05 33.12L168.05 33.12Q165.90 29.14 161.93 29.14Z');
      hello.getD({
        anchor: 'left middle'
      }, 'M5.27 23.70L5.27-26.72L10.62-26.72L10.62-6.64Q15.86-11.88 21.02-11.88L21.02-11.88Q26.89-11.88 29.60-6.71L29.60-6.71Q31.11-3.80 31.11 0.35L31.11 0.35L31.11 23.70L25.77 23.70L25.77 1.93Q25.77-6.79 20.18-6.79L20.18-6.79Q16.42-6.79 13.57-4.04L13.57-4.04Q10.62-1.09 10.62 2.71L10.62 2.71L10.62 23.70L5.27 23.70ZM62.05 13.08L67.68 13.08Q64.55 25.10 54.07 25.10L54.07 25.10Q47.57 25.10 43.77 19.69L43.77 19.69Q40.32 14.73 40.32 6.61L40.32 6.61Q40.32-1.16 43.56-6.12L43.56-6.12Q47.36-11.88 54-11.88L54-11.88Q66.97-11.88 67.82 7.70L67.82 7.70L45.74 7.70Q46.16 20.29 54.14 20.29L54.14 20.29Q60.47 20.29 62.05 13.08L62.05 13.08ZM45.95 3.09L62.05 3.09Q60.89-7.07 54-7.07L54-7.07Q47.36-7.07 45.95 3.09L45.95 3.09ZM92.81-26.72L92.81 15.82Q92.81 18.91 95.77 18.91L95.77 18.91Q98.19 18.91 101.07 18.35L101.07 18.35L101.07 23.73Q96.82 24.33 94.82 24.33L94.82 24.33Q87.19 24.33 87.19 16.84L87.19 16.84L87.19-26.72L92.81-26.72ZM128.81-26.72L128.81 15.82Q128.81 18.91 131.77 18.91L131.77 18.91Q134.19 18.91 137.07 18.35L137.07 18.35L137.07 23.73Q132.82 24.33 130.82 24.33L130.82 24.33Q123.19 24.33 123.19 16.84L123.19 16.84L123.19-26.72L128.81-26.72ZM162.07-11.88L162.07-11.88Q168.68-11.88 172.44-6.05L172.44-6.05Q175.68-1.20 175.68 6.61L175.68 6.61Q175.68 12.48 173.74 16.91L173.74 16.91Q170.16 25.14 161.93 25.14L161.93 25.14Q155.57 25.14 151.77 19.72L151.77 19.72Q148.32 14.77 148.32 6.61L148.32 6.61Q148.32-2.18 152.30-7.21L152.30-7.21Q156.09-11.88 162.07-11.88ZM161.93-6.86L161.93-6.86Q158.06-6.86 155.88-2.81L155.88-2.81Q153.95 0.74 153.95 6.61L153.95 6.61Q153.95 12.02 155.53 15.43L155.53 15.43Q157.71 20.11 162 20.11L162 20.11Q165.94 20.11 168.12 16.07L168.12 16.07Q170.05 12.52 170.05 6.68L170.05 6.68Q170.05 0.60 168.05-2.88L168.05-2.88Q165.90-6.86 161.93-6.86Z');
      hello.getD({
        anchor: 'left bottom'
      }, 'M5.27-12.30L5.27-62.72L10.62-62.72L10.62-42.64Q15.86-47.88 21.02-47.88L21.02-47.88Q26.89-47.88 29.60-42.71L29.60-42.71Q31.11-39.80 31.11-35.65L31.11-35.65L31.11-12.30L25.77-12.30L25.77-34.07Q25.77-42.79 20.18-42.79L20.18-42.79Q16.42-42.79 13.57-40.04L13.57-40.04Q10.62-37.09 10.62-33.29L10.62-33.29L10.62-12.30L5.27-12.30ZM62.05-22.92L67.68-22.92Q64.55-10.90 54.07-10.90L54.07-10.90Q47.57-10.90 43.77-16.31L43.77-16.31Q40.32-21.27 40.32-29.39L40.32-29.39Q40.32-37.16 43.56-42.12L43.56-42.12Q47.36-47.88 54-47.88L54-47.88Q66.97-47.88 67.82-28.30L67.82-28.30L45.74-28.30Q46.16-15.71 54.14-15.71L54.14-15.71Q60.47-15.71 62.05-22.92L62.05-22.92ZM45.95-32.91L62.05-32.91Q60.89-43.07 54-43.07L54-43.07Q47.36-43.07 45.95-32.91L45.95-32.91ZM92.81-62.72L92.81-20.18Q92.81-17.09 95.77-17.09L95.77-17.09Q98.19-17.09 101.07-17.65L101.07-17.65L101.07-12.27Q96.82-11.67 94.82-11.67L94.82-11.67Q87.19-11.67 87.19-19.16L87.19-19.16L87.19-62.72L92.81-62.72ZM128.81-62.72L128.81-20.18Q128.81-17.09 131.77-17.09L131.77-17.09Q134.19-17.09 137.07-17.65L137.07-17.65L137.07-12.27Q132.82-11.67 130.82-11.67L130.82-11.67Q123.19-11.67 123.19-19.16L123.19-19.16L123.19-62.72L128.81-62.72ZM162.07-47.88L162.07-47.88Q168.68-47.88 172.44-42.05L172.44-42.05Q175.68-37.20 175.68-29.39L175.68-29.39Q175.68-23.52 173.74-19.09L173.74-19.09Q170.16-10.86 161.93-10.86L161.93-10.86Q155.57-10.86 151.77-16.28L151.77-16.28Q148.32-21.23 148.32-29.39L148.32-29.39Q148.32-38.18 152.30-43.21L152.30-43.21Q156.09-47.88 162.07-47.88ZM161.93-42.86L161.93-42.86Q158.06-42.86 155.88-38.81L155.88-38.81Q153.95-35.26 153.95-29.39L153.95-29.39Q153.95-23.98 155.53-20.57L155.53-20.57Q157.71-15.89 162-15.89L162-15.89Q165.94-15.89 168.12-19.93L168.12-19.93Q170.05-23.48 170.05-29.32L170.05-29.32Q170.05-35.40 168.05-38.88L168.05-38.88Q165.90-42.86 161.93-42.86Z');
      hello.getD({
        anchor: 'center top'
      }, 'M-84.73 59.70L-84.73 9.28L-79.38 9.28L-79.38 29.36Q-74.14 24.12-68.98 24.12L-68.98 24.12Q-63.11 24.12-60.40 29.29L-60.40 29.29Q-58.89 32.20-58.89 36.35L-58.89 36.35L-58.89 59.70L-64.23 59.70L-64.23 37.93Q-64.23 29.21-69.82 29.21L-69.82 29.21Q-73.58 29.21-76.43 31.96L-76.43 31.96Q-79.38 34.91-79.38 38.71L-79.38 38.71L-79.38 59.70L-84.73 59.70ZM-27.95 49.08L-22.32 49.08Q-25.45 61.10-35.93 61.10L-35.93 61.10Q-42.43 61.10-46.23 55.69L-46.23 55.69Q-49.68 50.73-49.68 42.61L-49.68 42.61Q-49.68 34.84-46.44 29.88L-46.44 29.88Q-42.64 24.12-36 24.12L-36 24.12Q-23.03 24.12-22.18 43.70L-22.18 43.70L-44.26 43.70Q-43.84 56.29-35.86 56.29L-35.86 56.29Q-29.53 56.29-27.95 49.08L-27.95 49.08ZM-44.05 39.09L-27.95 39.09Q-29.11 28.93-36 28.93L-36 28.93Q-42.64 28.93-44.05 39.09L-44.05 39.09ZM2.81 9.28L2.81 51.82Q2.81 54.91 5.77 54.91L5.77 54.91Q8.19 54.91 11.07 54.35L11.07 54.35L11.07 59.73Q6.82 60.33 4.82 60.33L4.82 60.33Q-2.81 60.33-2.81 52.84L-2.81 52.84L-2.81 9.28L2.81 9.28ZM38.81 9.28L38.81 51.82Q38.81 54.91 41.77 54.91L41.77 54.91Q44.19 54.91 47.07 54.35L47.07 54.35L47.07 59.73Q42.82 60.33 40.82 60.33L40.82 60.33Q33.19 60.33 33.19 52.84L33.19 52.84L33.19 9.28L38.81 9.28ZM72.07 24.12L72.07 24.12Q78.68 24.12 82.44 29.95L82.44 29.95Q85.68 34.80 85.68 42.61L85.68 42.61Q85.68 48.48 83.74 52.91L83.74 52.91Q80.16 61.14 71.93 61.14L71.93 61.14Q65.57 61.14 61.77 55.72L61.77 55.72Q58.32 50.77 58.32 42.61L58.32 42.61Q58.32 33.82 62.30 28.79L62.30 28.79Q66.09 24.12 72.07 24.12ZM71.93 29.14L71.93 29.14Q68.06 29.14 65.88 33.19L65.88 33.19Q63.95 36.74 63.95 42.61L63.95 42.61Q63.95 48.02 65.53 51.43L65.53 51.43Q67.71 56.11 72 56.11L72 56.11Q75.94 56.11 78.12 52.07L78.12 52.07Q80.05 48.52 80.05 42.68L80.05 42.68Q80.05 36.60 78.05 33.12L78.05 33.12Q75.90 29.14 71.93 29.14Z');
      hello.getD({
        anchor: 'center middle'
      }, 'M-84.73 23.70L-84.73-26.72L-79.38-26.72L-79.38-6.64Q-74.14-11.88-68.98-11.88L-68.98-11.88Q-63.11-11.88-60.40-6.71L-60.40-6.71Q-58.89-3.80-58.89 0.35L-58.89 0.35L-58.89 23.70L-64.23 23.70L-64.23 1.93Q-64.23-6.79-69.82-6.79L-69.82-6.79Q-73.58-6.79-76.43-4.04L-76.43-4.04Q-79.38-1.09-79.38 2.71L-79.38 2.71L-79.38 23.70L-84.73 23.70ZM-27.95 13.08L-22.32 13.08Q-25.45 25.10-35.93 25.10L-35.93 25.10Q-42.43 25.10-46.23 19.69L-46.23 19.69Q-49.68 14.73-49.68 6.61L-49.68 6.61Q-49.68-1.16-46.44-6.12L-46.44-6.12Q-42.64-11.88-36-11.88L-36-11.88Q-23.03-11.88-22.18 7.70L-22.18 7.70L-44.26 7.70Q-43.84 20.29-35.86 20.29L-35.86 20.29Q-29.53 20.29-27.95 13.08L-27.95 13.08ZM-44.05 3.09L-27.95 3.09Q-29.11-7.07-36-7.07L-36-7.07Q-42.64-7.07-44.05 3.09L-44.05 3.09ZM2.81-26.72L2.81 15.82Q2.81 18.91 5.77 18.91L5.77 18.91Q8.19 18.91 11.07 18.35L11.07 18.35L11.07 23.73Q6.82 24.33 4.82 24.33L4.82 24.33Q-2.81 24.33-2.81 16.84L-2.81 16.84L-2.81-26.72L2.81-26.72ZM38.81-26.72L38.81 15.82Q38.81 18.91 41.77 18.91L41.77 18.91Q44.19 18.91 47.07 18.35L47.07 18.35L47.07 23.73Q42.82 24.33 40.82 24.33L40.82 24.33Q33.19 24.33 33.19 16.84L33.19 16.84L33.19-26.72L38.81-26.72ZM72.07-11.88L72.07-11.88Q78.68-11.88 82.44-6.05L82.44-6.05Q85.68-1.20 85.68 6.61L85.68 6.61Q85.68 12.48 83.74 16.91L83.74 16.91Q80.16 25.14 71.93 25.14L71.93 25.14Q65.57 25.14 61.77 19.72L61.77 19.72Q58.32 14.77 58.32 6.61L58.32 6.61Q58.32-2.18 62.30-7.21L62.30-7.21Q66.09-11.88 72.07-11.88ZM71.93-6.86L71.93-6.86Q68.06-6.86 65.88-2.81L65.88-2.81Q63.95 0.74 63.95 6.61L63.95 6.61Q63.95 12.02 65.53 15.43L65.53 15.43Q67.71 20.11 72 20.11L72 20.11Q75.94 20.11 78.12 16.07L78.12 16.07Q80.05 12.52 80.05 6.68L80.05 6.68Q80.05 0.60 78.05-2.88L78.05-2.88Q75.90-6.86 71.93-6.86Z');
      hello.getD({
        anchor: 'center bottom'
      }, 'M-84.73-12.30L-84.73-62.72L-79.38-62.72L-79.38-42.64Q-74.14-47.88-68.98-47.88L-68.98-47.88Q-63.11-47.88-60.40-42.71L-60.40-42.71Q-58.89-39.80-58.89-35.65L-58.89-35.65L-58.89-12.30L-64.23-12.30L-64.23-34.07Q-64.23-42.79-69.82-42.79L-69.82-42.79Q-73.58-42.79-76.43-40.04L-76.43-40.04Q-79.38-37.09-79.38-33.29L-79.38-33.29L-79.38-12.30L-84.73-12.30ZM-27.95-22.92L-22.32-22.92Q-25.45-10.90-35.93-10.90L-35.93-10.90Q-42.43-10.90-46.23-16.31L-46.23-16.31Q-49.68-21.27-49.68-29.39L-49.68-29.39Q-49.68-37.16-46.44-42.12L-46.44-42.12Q-42.64-47.88-36-47.88L-36-47.88Q-23.03-47.88-22.18-28.30L-22.18-28.30L-44.26-28.30Q-43.84-15.71-35.86-15.71L-35.86-15.71Q-29.53-15.71-27.95-22.92L-27.95-22.92ZM-44.05-32.91L-27.95-32.91Q-29.11-43.07-36-43.07L-36-43.07Q-42.64-43.07-44.05-32.91L-44.05-32.91ZM2.81-62.72L2.81-20.18Q2.81-17.09 5.77-17.09L5.77-17.09Q8.19-17.09 11.07-17.65L11.07-17.65L11.07-12.27Q6.82-11.67 4.82-11.67L4.82-11.67Q-2.81-11.67-2.81-19.16L-2.81-19.16L-2.81-62.72L2.81-62.72ZM38.81-62.72L38.81-20.18Q38.81-17.09 41.77-17.09L41.77-17.09Q44.19-17.09 47.07-17.65L47.07-17.65L47.07-12.27Q42.82-11.67 40.82-11.67L40.82-11.67Q33.19-11.67 33.19-19.16L33.19-19.16L33.19-62.72L38.81-62.72ZM72.07-47.88L72.07-47.88Q78.68-47.88 82.44-42.05L82.44-42.05Q85.68-37.20 85.68-29.39L85.68-29.39Q85.68-23.52 83.74-19.09L83.74-19.09Q80.16-10.86 71.93-10.86L71.93-10.86Q65.57-10.86 61.77-16.28L61.77-16.28Q58.32-21.23 58.32-29.39L58.32-29.39Q58.32-38.18 62.30-43.21L62.30-43.21Q66.09-47.88 72.07-47.88ZM71.93-42.86L71.93-42.86Q68.06-42.86 65.88-38.81L65.88-38.81Q63.95-35.26 63.95-29.39L63.95-29.39Q63.95-23.98 65.53-20.57L65.53-20.57Q67.71-15.89 72-15.89L72-15.89Q75.94-15.89 78.12-19.93L78.12-19.93Q80.05-23.48 80.05-29.32L80.05-29.32Q80.05-35.40 78.05-38.88L78.05-38.88Q75.90-42.86 71.93-42.86Z');
      hello.getD({
        anchor: 'right top'
      }, 'M-174.73 59.70L-174.73 9.28L-169.38 9.28L-169.38 29.36Q-164.14 24.12-158.98 24.12L-158.98 24.12Q-153.11 24.12-150.40 29.29L-150.40 29.29Q-148.89 32.20-148.89 36.35L-148.89 36.35L-148.89 59.70L-154.23 59.70L-154.23 37.93Q-154.23 29.21-159.82 29.21L-159.82 29.21Q-163.58 29.21-166.43 31.96L-166.43 31.96Q-169.38 34.91-169.38 38.71L-169.38 38.71L-169.38 59.70L-174.73 59.70ZM-117.95 49.08L-112.32 49.08Q-115.45 61.10-125.93 61.10L-125.93 61.10Q-132.43 61.10-136.23 55.69L-136.23 55.69Q-139.68 50.73-139.68 42.61L-139.68 42.61Q-139.68 34.84-136.44 29.88L-136.44 29.88Q-132.64 24.12-126 24.12L-126 24.12Q-113.03 24.12-112.18 43.70L-112.18 43.70L-134.26 43.70Q-133.84 56.29-125.86 56.29L-125.86 56.29Q-119.53 56.29-117.95 49.08L-117.95 49.08ZM-134.05 39.09L-117.95 39.09Q-119.11 28.93-126 28.93L-126 28.93Q-132.64 28.93-134.05 39.09L-134.05 39.09ZM-87.19 9.28L-87.19 51.82Q-87.19 54.91-84.23 54.91L-84.23 54.91Q-81.81 54.91-78.93 54.35L-78.93 54.35L-78.93 59.73Q-83.18 60.33-85.18 60.33L-85.18 60.33Q-92.81 60.33-92.81 52.84L-92.81 52.84L-92.81 9.28L-87.19 9.28ZM-51.19 9.28L-51.19 51.82Q-51.19 54.91-48.23 54.91L-48.23 54.91Q-45.81 54.91-42.93 54.35L-42.93 54.35L-42.93 59.73Q-47.18 60.33-49.18 60.33L-49.18 60.33Q-56.81 60.33-56.81 52.84L-56.81 52.84L-56.81 9.28L-51.19 9.28ZM-17.93 24.12L-17.93 24.12Q-11.32 24.12-7.56 29.95L-7.56 29.95Q-4.32 34.80-4.32 42.61L-4.32 42.61Q-4.32 48.48-6.26 52.91L-6.26 52.91Q-9.84 61.14-18.07 61.14L-18.07 61.14Q-24.43 61.14-28.23 55.72L-28.23 55.72Q-31.68 50.77-31.68 42.61L-31.68 42.61Q-31.68 33.82-27.70 28.79L-27.70 28.79Q-23.91 24.12-17.93 24.12ZM-18.07 29.14L-18.07 29.14Q-21.94 29.14-24.12 33.19L-24.12 33.19Q-26.05 36.74-26.05 42.61L-26.05 42.61Q-26.05 48.02-24.47 51.43L-24.47 51.43Q-22.29 56.11-18 56.11L-18 56.11Q-14.06 56.11-11.88 52.07L-11.88 52.07Q-9.95 48.52-9.95 42.68L-9.95 42.68Q-9.95 36.60-11.95 33.12L-11.95 33.12Q-14.10 29.14-18.07 29.14Z');
      hello.getD({
        anchor: 'right middle'
      }, 'M-174.73 23.70L-174.73-26.72L-169.38-26.72L-169.38-6.64Q-164.14-11.88-158.98-11.88L-158.98-11.88Q-153.11-11.88-150.40-6.71L-150.40-6.71Q-148.89-3.80-148.89 0.35L-148.89 0.35L-148.89 23.70L-154.23 23.70L-154.23 1.93Q-154.23-6.79-159.82-6.79L-159.82-6.79Q-163.58-6.79-166.43-4.04L-166.43-4.04Q-169.38-1.09-169.38 2.71L-169.38 2.71L-169.38 23.70L-174.73 23.70ZM-117.95 13.08L-112.32 13.08Q-115.45 25.10-125.93 25.10L-125.93 25.10Q-132.43 25.10-136.23 19.69L-136.23 19.69Q-139.68 14.73-139.68 6.61L-139.68 6.61Q-139.68-1.16-136.44-6.12L-136.44-6.12Q-132.64-11.88-126-11.88L-126-11.88Q-113.03-11.88-112.18 7.70L-112.18 7.70L-134.26 7.70Q-133.84 20.29-125.86 20.29L-125.86 20.29Q-119.53 20.29-117.95 13.08L-117.95 13.08ZM-134.05 3.09L-117.95 3.09Q-119.11-7.07-126-7.07L-126-7.07Q-132.64-7.07-134.05 3.09L-134.05 3.09ZM-87.19-26.72L-87.19 15.82Q-87.19 18.91-84.23 18.91L-84.23 18.91Q-81.81 18.91-78.93 18.35L-78.93 18.35L-78.93 23.73Q-83.18 24.33-85.18 24.33L-85.18 24.33Q-92.81 24.33-92.81 16.84L-92.81 16.84L-92.81-26.72L-87.19-26.72ZM-51.19-26.72L-51.19 15.82Q-51.19 18.91-48.23 18.91L-48.23 18.91Q-45.81 18.91-42.93 18.35L-42.93 18.35L-42.93 23.73Q-47.18 24.33-49.18 24.33L-49.18 24.33Q-56.81 24.33-56.81 16.84L-56.81 16.84L-56.81-26.72L-51.19-26.72ZM-17.93-11.88L-17.93-11.88Q-11.32-11.88-7.56-6.05L-7.56-6.05Q-4.32-1.20-4.32 6.61L-4.32 6.61Q-4.32 12.48-6.26 16.91L-6.26 16.91Q-9.84 25.14-18.07 25.14L-18.07 25.14Q-24.43 25.14-28.23 19.72L-28.23 19.72Q-31.68 14.77-31.68 6.61L-31.68 6.61Q-31.68-2.18-27.70-7.21L-27.70-7.21Q-23.91-11.88-17.93-11.88ZM-18.07-6.86L-18.07-6.86Q-21.94-6.86-24.12-2.81L-24.12-2.81Q-26.05 0.74-26.05 6.61L-26.05 6.61Q-26.05 12.02-24.47 15.43L-24.47 15.43Q-22.29 20.11-18 20.11L-18 20.11Q-14.06 20.11-11.88 16.07L-11.88 16.07Q-9.95 12.52-9.95 6.68L-9.95 6.68Q-9.95 0.60-11.95-2.88L-11.95-2.88Q-14.10-6.86-18.07-6.86Z');
      hello.getD({
        anchor: 'right bottom'
      }, 'M-174.73-12.30L-174.73-62.72L-169.38-62.72L-169.38-42.64Q-164.14-47.88-158.98-47.88L-158.98-47.88Q-153.11-47.88-150.40-42.71L-150.40-42.71Q-148.89-39.80-148.89-35.65L-148.89-35.65L-148.89-12.30L-154.23-12.30L-154.23-34.07Q-154.23-42.79-159.82-42.79L-159.82-42.79Q-163.58-42.79-166.43-40.04L-166.43-40.04Q-169.38-37.09-169.38-33.29L-169.38-33.29L-169.38-12.30L-174.73-12.30ZM-117.95-22.92L-112.32-22.92Q-115.45-10.90-125.93-10.90L-125.93-10.90Q-132.43-10.90-136.23-16.31L-136.23-16.31Q-139.68-21.27-139.68-29.39L-139.68-29.39Q-139.68-37.16-136.44-42.12L-136.44-42.12Q-132.64-47.88-126-47.88L-126-47.88Q-113.03-47.88-112.18-28.30L-112.18-28.30L-134.26-28.30Q-133.84-15.71-125.86-15.71L-125.86-15.71Q-119.53-15.71-117.95-22.92L-117.95-22.92ZM-134.05-32.91L-117.95-32.91Q-119.11-43.07-126-43.07L-126-43.07Q-132.64-43.07-134.05-32.91L-134.05-32.91ZM-87.19-62.72L-87.19-20.18Q-87.19-17.09-84.23-17.09L-84.23-17.09Q-81.81-17.09-78.93-17.65L-78.93-17.65L-78.93-12.27Q-83.18-11.67-85.18-11.67L-85.18-11.67Q-92.81-11.67-92.81-19.16L-92.81-19.16L-92.81-62.72L-87.19-62.72ZM-51.19-62.72L-51.19-20.18Q-51.19-17.09-48.23-17.09L-48.23-17.09Q-45.81-17.09-42.93-17.65L-42.93-17.65L-42.93-12.27Q-47.18-11.67-49.18-11.67L-49.18-11.67Q-56.81-11.67-56.81-19.16L-56.81-19.16L-56.81-62.72L-51.19-62.72ZM-17.93-47.88L-17.93-47.88Q-11.32-47.88-7.56-42.05L-7.56-42.05Q-4.32-37.20-4.32-29.39L-4.32-29.39Q-4.32-23.52-6.26-19.09L-6.26-19.09Q-9.84-10.86-18.07-10.86L-18.07-10.86Q-24.43-10.86-28.23-16.28L-28.23-16.28Q-31.68-21.23-31.68-29.39L-31.68-29.39Q-31.68-38.18-27.70-43.21L-27.70-43.21Q-23.91-47.88-17.93-47.88ZM-18.07-42.86L-18.07-42.86Q-21.94-42.86-24.12-38.81L-24.12-38.81Q-26.05-35.26-26.05-29.39L-26.05-29.39Q-26.05-23.98-24.47-20.57L-24.47-20.57Q-22.29-15.89-18-15.89L-18-15.89Q-14.06-15.89-11.88-19.93L-11.88-19.93Q-9.95-23.48-9.95-29.32L-9.95-29.32Q-9.95-35.40-11.95-38.88L-11.95-38.88Q-14.10-42.86-18.07-42.86Z');
      hello.getPath({
        x: 0,
        y: 0,
        fontSize: 72,
        anchor: 'top',
        attributes: {
          fill: 'red',
          stroke: 'black'
        }
      }, '<path fill="red" stroke="black" d="M5.27 59.70L5.27 9.28L10.62 9.28L10.62 29.36Q15.86 24.12 21.02 24.12L21.02 24.12Q26.89 24.12 29.60 29.29L29.60 29.29Q31.11 32.20 31.11 36.35L31.11 36.35L31.11 59.70L25.77 59.70L25.77 37.93Q25.77 29.21 20.18 29.21L20.18 29.21Q16.42 29.21 13.57 31.96L13.57 31.96Q10.62 34.91 10.62 38.71L10.62 38.71L10.62 59.70L5.27 59.70ZM62.05 49.08L67.68 49.08Q64.55 61.10 54.07 61.10L54.07 61.10Q47.57 61.10 43.77 55.69L43.77 55.69Q40.32 50.73 40.32 42.61L40.32 42.61Q40.32 34.84 43.56 29.88L43.56 29.88Q47.36 24.12 54 24.12L54 24.12Q66.97 24.12 67.82 43.70L67.82 43.70L45.74 43.70Q46.16 56.29 54.14 56.29L54.14 56.29Q60.47 56.29 62.05 49.08L62.05 49.08ZM45.95 39.09L62.05 39.09Q60.89 28.93 54 28.93L54 28.93Q47.36 28.93 45.95 39.09L45.95 39.09ZM92.81 9.28L92.81 51.82Q92.81 54.91 95.77 54.91L95.77 54.91Q98.19 54.91 101.07 54.35L101.07 54.35L101.07 59.73Q96.82 60.33 94.82 60.33L94.82 60.33Q87.19 60.33 87.19 52.84L87.19 52.84L87.19 9.28L92.81 9.28ZM128.81 9.28L128.81 51.82Q128.81 54.91 131.77 54.91L131.77 54.91Q134.19 54.91 137.07 54.35L137.07 54.35L137.07 59.73Q132.82 60.33 130.82 60.33L130.82 60.33Q123.19 60.33 123.19 52.84L123.19 52.84L123.19 9.28L128.81 9.28ZM162.07 24.12L162.07 24.12Q168.68 24.12 172.44 29.95L172.44 29.95Q175.68 34.80 175.68 42.61L175.68 42.61Q175.68 48.48 173.74 52.91L173.74 52.91Q170.16 61.14 161.93 61.14L161.93 61.14Q155.57 61.14 151.77 55.72L151.77 55.72Q148.32 50.77 148.32 42.61L148.32 42.61Q148.32 33.82 152.30 28.79L152.30 28.79Q156.09 24.12 162.07 24.12ZM161.93 29.14L161.93 29.14Q158.06 29.14 155.88 33.19L155.88 33.19Q153.95 36.74 153.95 42.61L153.95 42.61Q153.95 48.02 155.53 51.43L155.53 51.43Q157.71 56.11 162 56.11L162 56.11Q165.94 56.11 168.12 52.07L168.12 52.07Q170.05 48.52 170.05 42.68L170.05 42.68Q170.05 36.60 168.05 33.12L168.05 33.12Q165.90 29.14 161.93 29.14Z"/>');
      hello.getSVG({
        x: 0,
        y: 0,
        fontSize: 72,
        anchor: 'top',
        attributes: {
          fill: 'red',
          stroke: 'black'
        }
      }, '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="180" height="72"><path fill="red" stroke="black" d="M5.27 59.70L5.27 9.28L10.62 9.28L10.62 29.36Q15.86 24.12 21.02 24.12L21.02 24.12Q26.89 24.12 29.60 29.29L29.60 29.29Q31.11 32.20 31.11 36.35L31.11 36.35L31.11 59.70L25.77 59.70L25.77 37.93Q25.77 29.21 20.18 29.21L20.18 29.21Q16.42 29.21 13.57 31.96L13.57 31.96Q10.62 34.91 10.62 38.71L10.62 38.71L10.62 59.70L5.27 59.70ZM62.05 49.08L67.68 49.08Q64.55 61.10 54.07 61.10L54.07 61.10Q47.57 61.10 43.77 55.69L43.77 55.69Q40.32 50.73 40.32 42.61L40.32 42.61Q40.32 34.84 43.56 29.88L43.56 29.88Q47.36 24.12 54 24.12L54 24.12Q66.97 24.12 67.82 43.70L67.82 43.70L45.74 43.70Q46.16 56.29 54.14 56.29L54.14 56.29Q60.47 56.29 62.05 49.08L62.05 49.08ZM45.95 39.09L62.05 39.09Q60.89 28.93 54 28.93L54 28.93Q47.36 28.93 45.95 39.09L45.95 39.09ZM92.81 9.28L92.81 51.82Q92.81 54.91 95.77 54.91L95.77 54.91Q98.19 54.91 101.07 54.35L101.07 54.35L101.07 59.73Q96.82 60.33 94.82 60.33L94.82 60.33Q87.19 60.33 87.19 52.84L87.19 52.84L87.19 9.28L92.81 9.28ZM128.81 9.28L128.81 51.82Q128.81 54.91 131.77 54.91L131.77 54.91Q134.19 54.91 137.07 54.35L137.07 54.35L137.07 59.73Q132.82 60.33 130.82 60.33L130.82 60.33Q123.19 60.33 123.19 52.84L123.19 52.84L123.19 9.28L128.81 9.28ZM162.07 24.12L162.07 24.12Q168.68 24.12 172.44 29.95L172.44 29.95Q175.68 34.80 175.68 42.61L175.68 42.61Q175.68 48.48 173.74 52.91L173.74 52.91Q170.16 61.14 161.93 61.14L161.93 61.14Q155.57 61.14 151.77 55.72L151.77 55.72Q148.32 50.77 148.32 42.61L148.32 42.61Q148.32 33.82 152.30 28.79L152.30 28.79Q156.09 24.12 162.07 24.12ZM161.93 29.14L161.93 29.14Q158.06 29.14 155.88 33.19L155.88 33.19Q153.95 36.74 153.95 42.61L153.95 42.61Q153.95 48.02 155.53 51.43L155.53 51.43Q157.71 56.11 162 56.11L162 56.11Q165.94 56.11 168.12 52.07L168.12 52.07Q170.05 48.52 170.05 42.68L170.05 42.68Q170.05 36.60 168.05 33.12L168.05 33.12Q165.90 29.14 161.93 29.14Z"/></svg>');
    });
    describe('宇治', function () {
      var uji = new TextToSVGTest('宇治');
      uji.getD({}, 'M38.46-59.77L38.46-49.82L65.00-49.82L65.00-35.37L59.59-35.37L59.59-45.28L12.27-45.28L12.27-35.37L6.93-35.37L6.93-49.82L32.98-49.82L32.98-59.77L38.46-59.77ZM56.46-32.63L39.13-32.63L39.13-21.83L67.68-21.83L67.68-17.16L39.13-17.16L39.13 0Q39.13 3.06 37.62 4.22L37.62 4.22Q36.35 5.17 32.98 5.17L32.98 5.17Q28.44 5.17 22.82 4.54L22.82 4.54L22.04-0.81Q28.41 0.07 31.68 0.07L31.68 0.07Q33.79 0.07 33.79-1.72L33.79-1.72L33.79-17.16L4.39-17.16L4.39-21.83L33.79-21.83L33.79-32.63L15.47-32.63L15.47-37.30L56.46-37.30L56.46-32.63ZM101.57-34.07L101.57-34.07L102.23-35.12Q108.14-45.88 112.18-58.39L112.18-58.39L117.77-56.74Q112.96-44.19 107.47-34.77L107.47-34.77L107.19-34.28L107.93-34.28Q114.79-34.56 122.45-35.30L122.45-35.30L128.88-35.79Q125.30-40.75 120.87-45.28L120.87-45.28L125.12-47.81Q133.59-38.64 139.25-30.45L139.25-30.45L134.86-26.93Q132.47-30.59 131.77-31.68L131.77-31.68L129.80-31.46Q117.18-29.85 95.91-28.65L95.91-28.65L93.87-33.86Q99.67-33.96 101.57-34.07ZM99.91-23.70L133.49-23.70L133.49 5.03L128.39 5.03L128.39 1.16L105.01 1.16L105.01 5.03L99.91 5.03L99.91-23.70ZM128.39-19.30L105.01-19.30L105.01-3.30L128.39-3.30L128.39-19.30ZM95.70-46.51L91.79-42.19Q86.80-48.27 80.86-52.70L80.86-52.70L84.52-56.74Q90.14-52.70 95.70-46.51L95.70-46.51ZM92.00-29.88L88.10-25.56Q82.93-31.39 76.89-35.86L76.89-35.86L80.54-39.90Q87.01-35.30 92.00-29.88L92.00-29.88ZM81.42 4.46L77.17-0.14Q85.04-9.42 91.23-21.73L91.23-21.73L95.03-18.14Q88.63-4.89 81.42 4.46L81.42 4.46Z');
      uji.getD({
        x: -30
      }, 'M8.46-59.77L8.46-49.82L35.00-49.82L35.00-35.37L29.59-35.37L29.59-45.28L-17.73-45.28L-17.73-35.37L-23.07-35.37L-23.07-49.82L2.98-49.82L2.98-59.77L8.46-59.77ZM26.46-32.63L9.13-32.63L9.13-21.83L37.68-21.83L37.68-17.16L9.13-17.16L9.13 0Q9.13 3.06 7.62 4.22L7.62 4.22Q6.35 5.17 2.98 5.17L2.98 5.17Q-1.56 5.17-7.18 4.54L-7.18 4.54L-7.96-0.81Q-1.59 0.07 1.68 0.07L1.68 0.07Q3.79 0.07 3.79-1.72L3.79-1.72L3.79-17.16L-25.61-17.16L-25.61-21.83L3.79-21.83L3.79-32.63L-14.53-32.63L-14.53-37.30L26.46-37.30L26.46-32.63ZM71.57-34.07L71.57-34.07L72.23-35.12Q78.14-45.88 82.18-58.39L82.18-58.39L87.77-56.74Q82.96-44.19 77.47-34.77L77.47-34.77L77.19-34.28L77.93-34.28Q84.79-34.56 92.45-35.30L92.45-35.30L98.88-35.79Q95.30-40.75 90.87-45.28L90.87-45.28L95.12-47.81Q103.59-38.64 109.25-30.45L109.25-30.45L104.86-26.93Q102.47-30.59 101.77-31.68L101.77-31.68L99.80-31.46Q87.18-29.85 65.91-28.65L65.91-28.65L63.87-33.86Q69.67-33.96 71.57-34.07ZM69.91-23.70L103.49-23.70L103.49 5.03L98.39 5.03L98.39 1.16L75.01 1.16L75.01 5.03L69.91 5.03L69.91-23.70ZM98.39-19.30L75.01-19.30L75.01-3.30L98.39-3.30L98.39-19.30ZM65.70-46.51L61.79-42.19Q56.80-48.27 50.86-52.70L50.86-52.70L54.52-56.74Q60.14-52.70 65.70-46.51L65.70-46.51ZM62.00-29.88L58.10-25.56Q52.93-31.39 46.89-35.86L46.89-35.86L50.54-39.90Q57.01-35.30 62.00-29.88L62.00-29.88ZM51.42 4.46L47.17-0.14Q55.04-9.42 61.23-21.73L61.23-21.73L65.03-18.14Q58.63-4.89 51.42 4.46L51.42 4.46Z');
      uji.getD({
        x: 0
      }, 'M38.46-59.77L38.46-49.82L65.00-49.82L65.00-35.37L59.59-35.37L59.59-45.28L12.27-45.28L12.27-35.37L6.93-35.37L6.93-49.82L32.98-49.82L32.98-59.77L38.46-59.77ZM56.46-32.63L39.13-32.63L39.13-21.83L67.68-21.83L67.68-17.16L39.13-17.16L39.13 0Q39.13 3.06 37.62 4.22L37.62 4.22Q36.35 5.17 32.98 5.17L32.98 5.17Q28.44 5.17 22.82 4.54L22.82 4.54L22.04-0.81Q28.41 0.07 31.68 0.07L31.68 0.07Q33.79 0.07 33.79-1.72L33.79-1.72L33.79-17.16L4.39-17.16L4.39-21.83L33.79-21.83L33.79-32.63L15.47-32.63L15.47-37.30L56.46-37.30L56.46-32.63ZM101.57-34.07L101.57-34.07L102.23-35.12Q108.14-45.88 112.18-58.39L112.18-58.39L117.77-56.74Q112.96-44.19 107.47-34.77L107.47-34.77L107.19-34.28L107.93-34.28Q114.79-34.56 122.45-35.30L122.45-35.30L128.88-35.79Q125.30-40.75 120.87-45.28L120.87-45.28L125.12-47.81Q133.59-38.64 139.25-30.45L139.25-30.45L134.86-26.93Q132.47-30.59 131.77-31.68L131.77-31.68L129.80-31.46Q117.18-29.85 95.91-28.65L95.91-28.65L93.87-33.86Q99.67-33.96 101.57-34.07ZM99.91-23.70L133.49-23.70L133.49 5.03L128.39 5.03L128.39 1.16L105.01 1.16L105.01 5.03L99.91 5.03L99.91-23.70ZM128.39-19.30L105.01-19.30L105.01-3.30L128.39-3.30L128.39-19.30ZM95.70-46.51L91.79-42.19Q86.80-48.27 80.86-52.70L80.86-52.70L84.52-56.74Q90.14-52.70 95.70-46.51L95.70-46.51ZM92.00-29.88L88.10-25.56Q82.93-31.39 76.89-35.86L76.89-35.86L80.54-39.90Q87.01-35.30 92.00-29.88L92.00-29.88ZM81.42 4.46L77.17-0.14Q85.04-9.42 91.23-21.73L91.23-21.73L95.03-18.14Q88.63-4.89 81.42 4.46L81.42 4.46Z');
      uji.getD({
        x: 30
      }, 'M68.46-59.77L68.46-49.82L95.00-49.82L95.00-35.37L89.59-35.37L89.59-45.28L42.27-45.28L42.27-35.37L36.93-35.37L36.93-49.82L62.98-49.82L62.98-59.77L68.46-59.77ZM86.46-32.63L69.13-32.63L69.13-21.83L97.68-21.83L97.68-17.16L69.13-17.16L69.13 0Q69.13 3.06 67.62 4.22L67.62 4.22Q66.35 5.17 62.98 5.17L62.98 5.17Q58.44 5.17 52.82 4.54L52.82 4.54L52.04-0.81Q58.41 0.07 61.68 0.07L61.68 0.07Q63.79 0.07 63.79-1.72L63.79-1.72L63.79-17.16L34.39-17.16L34.39-21.83L63.79-21.83L63.79-32.63L45.47-32.63L45.47-37.30L86.46-37.30L86.46-32.63ZM131.57-34.07L131.57-34.07L132.23-35.12Q138.14-45.88 142.18-58.39L142.18-58.39L147.77-56.74Q142.96-44.19 137.47-34.77L137.47-34.77L137.19-34.28L137.93-34.28Q144.79-34.56 152.45-35.30L152.45-35.30L158.88-35.79Q155.30-40.75 150.87-45.28L150.87-45.28L155.12-47.81Q163.59-38.64 169.25-30.45L169.25-30.45L164.86-26.93Q162.47-30.59 161.77-31.68L161.77-31.68L159.80-31.46Q147.18-29.85 125.91-28.65L125.91-28.65L123.87-33.86Q129.67-33.96 131.57-34.07ZM129.91-23.70L163.49-23.70L163.49 5.03L158.39 5.03L158.39 1.16L135.01 1.16L135.01 5.03L129.91 5.03L129.91-23.70ZM158.39-19.30L135.01-19.30L135.01-3.30L158.39-3.30L158.39-19.30ZM125.70-46.51L121.79-42.19Q116.80-48.27 110.86-52.70L110.86-52.70L114.52-56.74Q120.14-52.70 125.70-46.51L125.70-46.51ZM122.00-29.88L118.10-25.56Q112.93-31.39 106.89-35.86L106.89-35.86L110.54-39.90Q117.01-35.30 122.00-29.88L122.00-29.88ZM111.42 4.46L107.17-0.14Q115.04-9.42 121.23-21.73L121.23-21.73L125.03-18.14Q118.63-4.89 111.42 4.46L111.42 4.46Z');
      uji.getD({
        y: -30
      }, 'M38.46-89.77L38.46-79.82L65.00-79.82L65.00-65.37L59.59-65.37L59.59-75.28L12.27-75.28L12.27-65.37L6.93-65.37L6.93-79.82L32.98-79.82L32.98-89.77L38.46-89.77ZM56.46-62.63L39.13-62.63L39.13-51.83L67.68-51.83L67.68-47.16L39.13-47.16L39.13-30Q39.13-26.94 37.62-25.78L37.62-25.78Q36.35-24.83 32.98-24.83L32.98-24.83Q28.44-24.83 22.82-25.46L22.82-25.46L22.04-30.81Q28.41-29.93 31.68-29.93L31.68-29.93Q33.79-29.93 33.79-31.72L33.79-31.72L33.79-47.16L4.39-47.16L4.39-51.83L33.79-51.83L33.79-62.63L15.47-62.63L15.47-67.30L56.46-67.30L56.46-62.63ZM101.57-64.07L101.57-64.07L102.23-65.12Q108.14-75.88 112.18-88.39L112.18-88.39L117.77-86.74Q112.96-74.19 107.47-64.77L107.47-64.77L107.19-64.28L107.93-64.28Q114.79-64.56 122.45-65.30L122.45-65.30L128.88-65.79Q125.30-70.75 120.87-75.28L120.87-75.28L125.12-77.81Q133.59-68.64 139.25-60.45L139.25-60.45L134.86-56.93Q132.47-60.59 131.77-61.68L131.77-61.68L129.80-61.46Q117.18-59.85 95.91-58.65L95.91-58.65L93.87-63.86Q99.67-63.96 101.57-64.07ZM99.91-53.70L133.49-53.70L133.49-24.97L128.39-24.97L128.39-28.84L105.01-28.84L105.01-24.97L99.91-24.97L99.91-53.70ZM128.39-49.30L105.01-49.30L105.01-33.30L128.39-33.30L128.39-49.30ZM95.70-76.51L91.79-72.19Q86.80-78.27 80.86-82.70L80.86-82.70L84.52-86.74Q90.14-82.70 95.70-76.51L95.70-76.51ZM92.00-59.88L88.10-55.56Q82.93-61.39 76.89-65.86L76.89-65.86L80.54-69.90Q87.01-65.30 92.00-59.88L92.00-59.88ZM81.42-25.54L77.17-30.14Q85.04-39.42 91.23-51.73L91.23-51.73L95.03-48.14Q88.63-34.89 81.42-25.54L81.42-25.54Z');
      uji.getD({
        y: 0
      }, 'M38.46-59.77L38.46-49.82L65.00-49.82L65.00-35.37L59.59-35.37L59.59-45.28L12.27-45.28L12.27-35.37L6.93-35.37L6.93-49.82L32.98-49.82L32.98-59.77L38.46-59.77ZM56.46-32.63L39.13-32.63L39.13-21.83L67.68-21.83L67.68-17.16L39.13-17.16L39.13 0Q39.13 3.06 37.62 4.22L37.62 4.22Q36.35 5.17 32.98 5.17L32.98 5.17Q28.44 5.17 22.82 4.54L22.82 4.54L22.04-0.81Q28.41 0.07 31.68 0.07L31.68 0.07Q33.79 0.07 33.79-1.72L33.79-1.72L33.79-17.16L4.39-17.16L4.39-21.83L33.79-21.83L33.79-32.63L15.47-32.63L15.47-37.30L56.46-37.30L56.46-32.63ZM101.57-34.07L101.57-34.07L102.23-35.12Q108.14-45.88 112.18-58.39L112.18-58.39L117.77-56.74Q112.96-44.19 107.47-34.77L107.47-34.77L107.19-34.28L107.93-34.28Q114.79-34.56 122.45-35.30L122.45-35.30L128.88-35.79Q125.30-40.75 120.87-45.28L120.87-45.28L125.12-47.81Q133.59-38.64 139.25-30.45L139.25-30.45L134.86-26.93Q132.47-30.59 131.77-31.68L131.77-31.68L129.80-31.46Q117.18-29.85 95.91-28.65L95.91-28.65L93.87-33.86Q99.67-33.96 101.57-34.07ZM99.91-23.70L133.49-23.70L133.49 5.03L128.39 5.03L128.39 1.16L105.01 1.16L105.01 5.03L99.91 5.03L99.91-23.70ZM128.39-19.30L105.01-19.30L105.01-3.30L128.39-3.30L128.39-19.30ZM95.70-46.51L91.79-42.19Q86.80-48.27 80.86-52.70L80.86-52.70L84.52-56.74Q90.14-52.70 95.70-46.51L95.70-46.51ZM92.00-29.88L88.10-25.56Q82.93-31.39 76.89-35.86L76.89-35.86L80.54-39.90Q87.01-35.30 92.00-29.88L92.00-29.88ZM81.42 4.46L77.17-0.14Q85.04-9.42 91.23-21.73L91.23-21.73L95.03-18.14Q88.63-4.89 81.42 4.46L81.42 4.46Z');
      uji.getD({
        y: 30
      }, 'M38.46-29.77L38.46-19.82L65.00-19.82L65.00-5.37L59.59-5.37L59.59-15.28L12.27-15.28L12.27-5.37L6.93-5.37L6.93-19.82L32.98-19.82L32.98-29.77L38.46-29.77ZM56.46-2.63L39.13-2.63L39.13 8.17L67.68 8.17L67.68 12.84L39.13 12.84L39.13 30Q39.13 33.06 37.62 34.22L37.62 34.22Q36.35 35.17 32.98 35.17L32.98 35.17Q28.44 35.17 22.82 34.54L22.82 34.54L22.04 29.19Q28.41 30.07 31.68 30.07L31.68 30.07Q33.79 30.07 33.79 28.28L33.79 28.28L33.79 12.84L4.39 12.84L4.39 8.17L33.79 8.17L33.79-2.63L15.47-2.63L15.47-7.30L56.46-7.30L56.46-2.63ZM101.57-4.07L101.57-4.07L102.23-5.12Q108.14-15.88 112.18-28.39L112.18-28.39L117.77-26.74Q112.96-14.19 107.47-4.77L107.47-4.77L107.19-4.28L107.93-4.28Q114.79-4.56 122.45-5.30L122.45-5.30L128.88-5.79Q125.30-10.75 120.87-15.28L120.87-15.28L125.12-17.81Q133.59-8.64 139.25-0.45L139.25-0.45L134.86 3.07Q132.47-0.59 131.77-1.68L131.77-1.68L129.80-1.46Q117.18 0.15 95.91 1.35L95.91 1.35L93.87-3.86Q99.67-3.96 101.57-4.07ZM99.91 6.30L133.49 6.30L133.49 35.03L128.39 35.03L128.39 31.16L105.01 31.16L105.01 35.03L99.91 35.03L99.91 6.30ZM128.39 10.70L105.01 10.70L105.01 26.70L128.39 26.70L128.39 10.70ZM95.70-16.51L91.79-12.19Q86.80-18.27 80.86-22.70L80.86-22.70L84.52-26.74Q90.14-22.70 95.70-16.51L95.70-16.51ZM92.00 0.12L88.10 4.44Q82.93-1.39 76.89-5.86L76.89-5.86L80.54-9.90Q87.01-5.30 92.00 0.12L92.00 0.12ZM81.42 34.46L77.17 29.86Q85.04 20.58 91.23 8.27L91.23 8.27L95.03 11.86Q88.63 25.11 81.42 34.46L81.42 34.46Z');
      uji.getD({
        fontSize: 10
      }, 'M5.34-8.30L5.34-6.92L9.03-6.92L9.03-4.91L8.28-4.91L8.28-6.29L1.70-6.29L1.70-4.91L0.96-4.91L0.96-6.92L4.58-6.92L4.58-8.30L5.34-8.30ZM7.84-4.53L5.43-4.53L5.43-3.03L9.40-3.03L9.40-2.38L5.43-2.38L5.43 0Q5.43 0.42 5.22 0.59L5.22 0.59Q5.05 0.72 4.58 0.72L4.58 0.72Q3.95 0.72 3.17 0.63L3.17 0.63L3.06-0.11Q3.95 0.01 4.40 0.01L4.40 0.01Q4.69 0.01 4.69-0.24L4.69-0.24L4.69-2.38L0.61-2.38L0.61-3.03L4.69-3.03L4.69-4.53L2.15-4.53L2.15-5.18L7.84-5.18L7.84-4.53ZM14.11-4.73L14.11-4.73L14.20-4.88Q15.02-6.37 15.58-8.11L15.58-8.11L16.36-7.88Q15.69-6.14 14.93-4.83L14.93-4.83L14.89-4.76L14.99-4.76Q15.94-4.80 17.01-4.90L17.01-4.90L17.90-4.97Q17.40-5.66 16.79-6.29L16.79-6.29L17.38-6.64Q18.55-5.37 19.34-4.23L19.34-4.23L18.73-3.74Q18.40-4.25 18.30-4.40L18.30-4.40L18.03-4.37Q16.27-4.15 13.32-3.98L13.32-3.98L13.04-4.70Q13.84-4.72 14.11-4.73ZM13.88-3.29L18.54-3.29L18.54 0.70L17.83 0.70L17.83 0.16L14.58 0.16L14.58 0.70L13.88 0.70L13.88-3.29ZM17.83-2.68L14.58-2.68L14.58-0.46L17.83-0.46L17.83-2.68ZM13.29-6.46L12.75-5.86Q12.06-6.70 11.23-7.32L11.23-7.32L11.74-7.88Q12.52-7.32 13.29-6.46L13.29-6.46ZM12.78-4.15L12.24-3.55Q11.52-4.36 10.68-4.98L10.68-4.98L11.19-5.54Q12.08-4.90 12.78-4.15L12.78-4.15ZM11.31 0.62L10.72-0.02Q11.81-1.31 12.67-3.02L12.67-3.02L13.20-2.52Q12.31-0.68 11.31 0.62L11.31 0.62Z');
      uji.getD({
        fontSize: 30
      }, 'M16.03-24.90L16.03-20.76L27.08-20.76L27.08-14.74L24.83-14.74L24.83-18.87L5.11-18.87L5.11-14.74L2.89-14.74L2.89-20.76L13.74-20.76L13.74-24.90L16.03-24.90ZM23.53-13.59L16.30-13.59L16.30-9.10L28.20-9.10L28.20-7.15L16.30-7.15L16.30 0Q16.30 1.27 15.67 1.76L15.67 1.76Q15.15 2.15 13.74 2.15L13.74 2.15Q11.85 2.15 9.51 1.89L9.51 1.89L9.18-0.34Q11.84 0.03 13.20 0.03L13.20 0.03Q14.08 0.03 14.08-0.72L14.08-0.72L14.08-7.15L1.83-7.15L1.83-9.10L14.08-9.10L14.08-13.59L6.45-13.59L6.45-15.54L23.53-15.54L23.53-13.59ZM42.32-14.19L42.32-14.19L42.60-14.63Q45.06-19.12 46.74-24.33L46.74-24.33L49.07-23.64Q47.07-18.41 44.78-14.49L44.78-14.49L44.66-14.28L44.97-14.28Q47.83-14.40 51.02-14.71L51.02-14.71L53.70-14.91Q52.21-16.98 50.36-18.87L50.36-18.87L52.13-19.92Q55.66-16.10 58.02-12.69L58.02-12.69L56.19-11.22Q55.20-12.74 54.90-13.20L54.90-13.20L54.08-13.11Q48.82-12.44 39.96-11.94L39.96-11.94L39.11-14.11Q41.53-14.15 42.32-14.19ZM41.63-9.87L55.62-9.87L55.62 2.09L53.50 2.09L53.50 0.48L43.75 0.48L43.75 2.09L41.63 2.09L41.63-9.87ZM53.50-8.04L43.75-8.04L43.75-1.38L53.50-1.38L53.50-8.04ZM39.87-19.38L38.25-17.58Q36.17-20.11 33.69-21.96L33.69-21.96L35.21-23.64Q37.56-21.96 39.87-19.38L39.87-19.38ZM38.33-12.45L36.71-10.65Q34.56-13.08 32.04-14.94L32.04-14.94L33.56-16.63Q36.25-14.71 38.33-12.45L38.33-12.45ZM33.93 1.86L32.15-0.06Q35.43-3.93 38.01-9.05L38.01-9.05L39.59-7.56Q36.93-2.04 33.93 1.86L33.93 1.86Z');
      uji.getD({
        kerning: false
      }, 'M38.46-59.77L38.46-49.82L65.00-49.82L65.00-35.37L59.59-35.37L59.59-45.28L12.27-45.28L12.27-35.37L6.93-35.37L6.93-49.82L32.98-49.82L32.98-59.77L38.46-59.77ZM56.46-32.63L39.13-32.63L39.13-21.83L67.68-21.83L67.68-17.16L39.13-17.16L39.13 0Q39.13 3.06 37.62 4.22L37.62 4.22Q36.35 5.17 32.98 5.17L32.98 5.17Q28.44 5.17 22.82 4.54L22.82 4.54L22.04-0.81Q28.41 0.07 31.68 0.07L31.68 0.07Q33.79 0.07 33.79-1.72L33.79-1.72L33.79-17.16L4.39-17.16L4.39-21.83L33.79-21.83L33.79-32.63L15.47-32.63L15.47-37.30L56.46-37.30L56.46-32.63ZM101.57-34.07L101.57-34.07L102.23-35.12Q108.14-45.88 112.18-58.39L112.18-58.39L117.77-56.74Q112.96-44.19 107.47-34.77L107.47-34.77L107.19-34.28L107.93-34.28Q114.79-34.56 122.45-35.30L122.45-35.30L128.88-35.79Q125.30-40.75 120.87-45.28L120.87-45.28L125.12-47.81Q133.59-38.64 139.25-30.45L139.25-30.45L134.86-26.93Q132.47-30.59 131.77-31.68L131.77-31.68L129.80-31.46Q117.18-29.85 95.91-28.65L95.91-28.65L93.87-33.86Q99.67-33.96 101.57-34.07ZM99.91-23.70L133.49-23.70L133.49 5.03L128.39 5.03L128.39 1.16L105.01 1.16L105.01 5.03L99.91 5.03L99.91-23.70ZM128.39-19.30L105.01-19.30L105.01-3.30L128.39-3.30L128.39-19.30ZM95.70-46.51L91.79-42.19Q86.80-48.27 80.86-52.70L80.86-52.70L84.52-56.74Q90.14-52.70 95.70-46.51L95.70-46.51ZM92.00-29.88L88.10-25.56Q82.93-31.39 76.89-35.86L76.89-35.86L80.54-39.90Q87.01-35.30 92.00-29.88L92.00-29.88ZM81.42 4.46L77.17-0.14Q85.04-9.42 91.23-21.73L91.23-21.73L95.03-18.14Q88.63-4.89 81.42 4.46L81.42 4.46Z');
      uji.getD({
        kerning: true
      }, 'M38.46-59.77L38.46-49.82L65.00-49.82L65.00-35.37L59.59-35.37L59.59-45.28L12.27-45.28L12.27-35.37L6.93-35.37L6.93-49.82L32.98-49.82L32.98-59.77L38.46-59.77ZM56.46-32.63L39.13-32.63L39.13-21.83L67.68-21.83L67.68-17.16L39.13-17.16L39.13 0Q39.13 3.06 37.62 4.22L37.62 4.22Q36.35 5.17 32.98 5.17L32.98 5.17Q28.44 5.17 22.82 4.54L22.82 4.54L22.04-0.81Q28.41 0.07 31.68 0.07L31.68 0.07Q33.79 0.07 33.79-1.72L33.79-1.72L33.79-17.16L4.39-17.16L4.39-21.83L33.79-21.83L33.79-32.63L15.47-32.63L15.47-37.30L56.46-37.30L56.46-32.63ZM101.57-34.07L101.57-34.07L102.23-35.12Q108.14-45.88 112.18-58.39L112.18-58.39L117.77-56.74Q112.96-44.19 107.47-34.77L107.47-34.77L107.19-34.28L107.93-34.28Q114.79-34.56 122.45-35.30L122.45-35.30L128.88-35.79Q125.30-40.75 120.87-45.28L120.87-45.28L125.12-47.81Q133.59-38.64 139.25-30.45L139.25-30.45L134.86-26.93Q132.47-30.59 131.77-31.68L131.77-31.68L129.80-31.46Q117.18-29.85 95.91-28.65L95.91-28.65L93.87-33.86Q99.67-33.96 101.57-34.07ZM99.91-23.70L133.49-23.70L133.49 5.03L128.39 5.03L128.39 1.16L105.01 1.16L105.01 5.03L99.91 5.03L99.91-23.70ZM128.39-19.30L105.01-19.30L105.01-3.30L128.39-3.30L128.39-19.30ZM95.70-46.51L91.79-42.19Q86.80-48.27 80.86-52.70L80.86-52.70L84.52-56.74Q90.14-52.70 95.70-46.51L95.70-46.51ZM92.00-29.88L88.10-25.56Q82.93-31.39 76.89-35.86L76.89-35.86L80.54-39.90Q87.01-35.30 92.00-29.88L92.00-29.88ZM81.42 4.46L77.17-0.14Q85.04-9.42 91.23-21.73L91.23-21.73L95.03-18.14Q88.63-4.89 81.42 4.46L81.42 4.46Z');
      uji.getD({
        letterSpacing: 0.2
      }, 'M38.46-59.77L38.46-49.82L65.00-49.82L65.00-35.37L59.59-35.37L59.59-45.28L12.27-45.28L12.27-35.37L6.93-35.37L6.93-49.82L32.98-49.82L32.98-59.77L38.46-59.77ZM56.46-32.63L39.13-32.63L39.13-21.83L67.68-21.83L67.68-17.16L39.13-17.16L39.13 0Q39.13 3.06 37.62 4.22L37.62 4.22Q36.35 5.17 32.98 5.17L32.98 5.17Q28.44 5.17 22.82 4.54L22.82 4.54L22.04-0.81Q28.41 0.07 31.68 0.07L31.68 0.07Q33.79 0.07 33.79-1.72L33.79-1.72L33.79-17.16L4.39-17.16L4.39-21.83L33.79-21.83L33.79-32.63L15.47-32.63L15.47-37.30L56.46-37.30L56.46-32.63ZM115.97-34.07L115.97-34.07L116.63-35.12Q122.54-45.88 126.58-58.39L126.58-58.39L132.17-56.74Q127.36-44.19 121.87-34.77L121.87-34.77L121.59-34.28L122.33-34.28Q129.19-34.56 136.85-35.30L136.85-35.30L143.28-35.79Q139.70-40.75 135.27-45.28L135.27-45.28L139.52-47.81Q147.99-38.64 153.65-30.45L153.65-30.45L149.26-26.93Q146.87-30.59 146.17-31.68L146.17-31.68L144.20-31.46Q131.58-29.85 110.31-28.65L110.31-28.65L108.27-33.86Q114.07-33.96 115.97-34.07ZM114.31-23.70L147.89-23.70L147.89 5.03L142.79 5.03L142.79 1.16L119.41 1.16L119.41 5.03L114.31 5.03L114.31-23.70ZM142.79-19.30L119.41-19.30L119.41-3.30L142.79-3.30L142.79-19.30ZM110.10-46.51L106.19-42.19Q101.20-48.27 95.26-52.70L95.26-52.70L98.92-56.74Q104.54-52.70 110.10-46.51L110.10-46.51ZM106.40-29.88L102.50-25.56Q97.33-31.39 91.29-35.86L91.29-35.86L94.94-39.90Q101.41-35.30 106.40-29.88L106.40-29.88ZM95.82 4.46L91.57-0.14Q99.44-9.42 105.63-21.73L105.63-21.73L109.43-18.14Q103.03-4.89 95.82 4.46L95.82 4.46Z');
      uji.getD({
        tracking: 200
      }, 'M38.46-59.77L38.46-49.82L65.00-49.82L65.00-35.37L59.59-35.37L59.59-45.28L12.27-45.28L12.27-35.37L6.93-35.37L6.93-49.82L32.98-49.82L32.98-59.77L38.46-59.77ZM56.46-32.63L39.13-32.63L39.13-21.83L67.68-21.83L67.68-17.16L39.13-17.16L39.13 0Q39.13 3.06 37.62 4.22L37.62 4.22Q36.35 5.17 32.98 5.17L32.98 5.17Q28.44 5.17 22.82 4.54L22.82 4.54L22.04-0.81Q28.41 0.07 31.68 0.07L31.68 0.07Q33.79 0.07 33.79-1.72L33.79-1.72L33.79-17.16L4.39-17.16L4.39-21.83L33.79-21.83L33.79-32.63L15.47-32.63L15.47-37.30L56.46-37.30L56.46-32.63ZM115.97-34.07L115.97-34.07L116.63-35.12Q122.54-45.88 126.58-58.39L126.58-58.39L132.17-56.74Q127.36-44.19 121.87-34.77L121.87-34.77L121.59-34.28L122.33-34.28Q129.19-34.56 136.85-35.30L136.85-35.30L143.28-35.79Q139.70-40.75 135.27-45.28L135.27-45.28L139.52-47.81Q147.99-38.64 153.65-30.45L153.65-30.45L149.26-26.93Q146.87-30.59 146.17-31.68L146.17-31.68L144.20-31.46Q131.58-29.85 110.31-28.65L110.31-28.65L108.27-33.86Q114.07-33.96 115.97-34.07ZM114.31-23.70L147.89-23.70L147.89 5.03L142.79 5.03L142.79 1.16L119.41 1.16L119.41 5.03L114.31 5.03L114.31-23.70ZM142.79-19.30L119.41-19.30L119.41-3.30L142.79-3.30L142.79-19.30ZM110.10-46.51L106.19-42.19Q101.20-48.27 95.26-52.70L95.26-52.70L98.92-56.74Q104.54-52.70 110.10-46.51L110.10-46.51ZM106.40-29.88L102.50-25.56Q97.33-31.39 91.29-35.86L91.29-35.86L94.94-39.90Q101.41-35.30 106.40-29.88L106.40-29.88ZM95.82 4.46L91.57-0.14Q99.44-9.42 105.63-21.73L105.63-21.73L109.43-18.14Q103.03-4.89 95.82 4.46L95.82 4.46Z');
      uji.getD({
        anchor: 'left'
      }, 'M38.46-59.77L38.46-49.82L65.00-49.82L65.00-35.37L59.59-35.37L59.59-45.28L12.27-45.28L12.27-35.37L6.93-35.37L6.93-49.82L32.98-49.82L32.98-59.77L38.46-59.77ZM56.46-32.63L39.13-32.63L39.13-21.83L67.68-21.83L67.68-17.16L39.13-17.16L39.13 0Q39.13 3.06 37.62 4.22L37.62 4.22Q36.35 5.17 32.98 5.17L32.98 5.17Q28.44 5.17 22.82 4.54L22.82 4.54L22.04-0.81Q28.41 0.07 31.68 0.07L31.68 0.07Q33.79 0.07 33.79-1.72L33.79-1.72L33.79-17.16L4.39-17.16L4.39-21.83L33.79-21.83L33.79-32.63L15.47-32.63L15.47-37.30L56.46-37.30L56.46-32.63ZM101.57-34.07L101.57-34.07L102.23-35.12Q108.14-45.88 112.18-58.39L112.18-58.39L117.77-56.74Q112.96-44.19 107.47-34.77L107.47-34.77L107.19-34.28L107.93-34.28Q114.79-34.56 122.45-35.30L122.45-35.30L128.88-35.79Q125.30-40.75 120.87-45.28L120.87-45.28L125.12-47.81Q133.59-38.64 139.25-30.45L139.25-30.45L134.86-26.93Q132.47-30.59 131.77-31.68L131.77-31.68L129.80-31.46Q117.18-29.85 95.91-28.65L95.91-28.65L93.87-33.86Q99.67-33.96 101.57-34.07ZM99.91-23.70L133.49-23.70L133.49 5.03L128.39 5.03L128.39 1.16L105.01 1.16L105.01 5.03L99.91 5.03L99.91-23.70ZM128.39-19.30L105.01-19.30L105.01-3.30L128.39-3.30L128.39-19.30ZM95.70-46.51L91.79-42.19Q86.80-48.27 80.86-52.70L80.86-52.70L84.52-56.74Q90.14-52.70 95.70-46.51L95.70-46.51ZM92.00-29.88L88.10-25.56Q82.93-31.39 76.89-35.86L76.89-35.86L80.54-39.90Q87.01-35.30 92.00-29.88L92.00-29.88ZM81.42 4.46L77.17-0.14Q85.04-9.42 91.23-21.73L91.23-21.73L95.03-18.14Q88.63-4.89 81.42 4.46L81.42 4.46Z');
      uji.getD({
        anchor: 'center'
      }, 'M-33.54-59.77L-33.54-49.82L-7.00-49.82L-7.00-35.37L-12.41-35.37L-12.41-45.28L-59.73-45.28L-59.73-35.37L-65.07-35.37L-65.07-49.82L-39.02-49.82L-39.02-59.77L-33.54-59.77ZM-15.54-32.63L-32.87-32.63L-32.87-21.83L-4.32-21.83L-4.32-17.16L-32.87-17.16L-32.87 0Q-32.87 3.06-34.38 4.22L-34.38 4.22Q-35.65 5.17-39.02 5.17L-39.02 5.17Q-43.56 5.17-49.18 4.54L-49.18 4.54L-49.96-0.81Q-43.59 0.07-40.32 0.07L-40.32 0.07Q-38.21 0.07-38.21-1.72L-38.21-1.72L-38.21-17.16L-67.61-17.16L-67.61-21.83L-38.21-21.83L-38.21-32.63L-56.53-32.63L-56.53-37.30L-15.54-37.30L-15.54-32.63ZM29.57-34.07L29.57-34.07L30.23-35.12Q36.14-45.88 40.18-58.39L40.18-58.39L45.77-56.74Q40.96-44.19 35.47-34.77L35.47-34.77L35.19-34.28L35.93-34.28Q42.79-34.56 50.45-35.30L50.45-35.30L56.88-35.79Q53.30-40.75 48.87-45.28L48.87-45.28L53.12-47.81Q61.59-38.64 67.25-30.45L67.25-30.45L62.86-26.93Q60.47-30.59 59.77-31.68L59.77-31.68L57.80-31.46Q45.18-29.85 23.91-28.65L23.91-28.65L21.87-33.86Q27.67-33.96 29.57-34.07ZM27.91-23.70L61.49-23.70L61.49 5.03L56.39 5.03L56.39 1.16L33.01 1.16L33.01 5.03L27.91 5.03L27.91-23.70ZM56.39-19.30L33.01-19.30L33.01-3.30L56.39-3.30L56.39-19.30ZM23.70-46.51L19.79-42.19Q14.80-48.27 8.86-52.70L8.86-52.70L12.52-56.74Q18.14-52.70 23.70-46.51L23.70-46.51ZM20.00-29.88L16.10-25.56Q10.93-31.39 4.89-35.86L4.89-35.86L8.54-39.90Q15.01-35.30 20.00-29.88L20.00-29.88ZM9.42 4.46L5.17-0.14Q13.04-9.42 19.23-21.73L19.23-21.73L23.03-18.14Q16.63-4.89 9.42 4.46L9.42 4.46Z');
      uji.getD({
        anchor: 'right'
      }, 'M-105.54-59.77L-105.54-49.82L-79.00-49.82L-79.00-35.37L-84.41-35.37L-84.41-45.28L-131.73-45.28L-131.73-35.37L-137.07-35.37L-137.07-49.82L-111.02-49.82L-111.02-59.77L-105.54-59.77ZM-87.54-32.63L-104.87-32.63L-104.87-21.83L-76.32-21.83L-76.32-17.16L-104.87-17.16L-104.87 0Q-104.87 3.06-106.38 4.22L-106.38 4.22Q-107.65 5.17-111.02 5.17L-111.02 5.17Q-115.56 5.17-121.18 4.54L-121.18 4.54L-121.96-0.81Q-115.59 0.07-112.32 0.07L-112.32 0.07Q-110.21 0.07-110.21-1.72L-110.21-1.72L-110.21-17.16L-139.61-17.16L-139.61-21.83L-110.21-21.83L-110.21-32.63L-128.53-32.63L-128.53-37.30L-87.54-37.30L-87.54-32.63ZM-42.43-34.07L-42.43-34.07L-41.77-35.12Q-35.86-45.88-31.82-58.39L-31.82-58.39L-26.23-56.74Q-31.04-44.19-36.53-34.77L-36.53-34.77L-36.81-34.28L-36.07-34.28Q-29.21-34.56-21.55-35.30L-21.55-35.30L-15.12-35.79Q-18.70-40.75-23.13-45.28L-23.13-45.28L-18.88-47.81Q-10.41-38.64-4.75-30.45L-4.75-30.45L-9.14-26.93Q-11.53-30.59-12.23-31.68L-12.23-31.68L-14.20-31.46Q-26.82-29.85-48.09-28.65L-48.09-28.65L-50.13-33.86Q-44.33-33.96-42.43-34.07ZM-44.09-23.70L-10.51-23.70L-10.51 5.03L-15.61 5.03L-15.61 1.16L-38.99 1.16L-38.99 5.03L-44.09 5.03L-44.09-23.70ZM-15.61-19.30L-38.99-19.30L-38.99-3.30L-15.61-3.30L-15.61-19.30ZM-48.30-46.51L-52.21-42.19Q-57.20-48.27-63.14-52.70L-63.14-52.70L-59.48-56.74Q-53.86-52.70-48.30-46.51L-48.30-46.51ZM-52.00-29.88L-55.90-25.56Q-61.07-31.39-67.11-35.86L-67.11-35.86L-63.46-39.90Q-56.99-35.30-52.00-29.88L-52.00-29.88ZM-62.58 4.46L-66.83-0.14Q-58.96-9.42-52.77-21.73L-52.77-21.73L-48.97-18.14Q-55.37-4.89-62.58 4.46L-62.58 4.46Z');
      uji.getD({
        anchor: 'top'
      }, 'M38.46 3.59L38.46 13.54L65.00 13.54L65.00 27.98L59.59 27.98L59.59 18.07L12.27 18.07L12.27 27.98L6.93 27.98L6.93 13.54L32.98 13.54L32.98 3.59L38.46 3.59ZM56.46 30.73L39.13 30.73L39.13 41.52L67.68 41.52L67.68 46.20L39.13 46.20L39.13 63.35Q39.13 66.41 37.62 67.57L37.62 67.57Q36.35 68.52 32.98 68.52L32.98 68.52Q28.44 68.52 22.82 67.89L22.82 67.89L22.04 62.54Q28.41 63.42 31.68 63.42L31.68 63.42Q33.79 63.42 33.79 61.63L33.79 61.63L33.79 46.20L4.39 46.20L4.39 41.52L33.79 41.52L33.79 30.73L15.47 30.73L15.47 26.05L56.46 26.05L56.46 30.73ZM101.57 29.29L101.57 29.29L102.23 28.23Q108.14 17.47 112.18 4.96L112.18 4.96L117.77 6.61Q112.96 19.16 107.47 28.58L107.47 28.58L107.19 29.07L107.93 29.07Q114.79 28.79 122.45 28.05L122.45 28.05L128.88 27.56Q125.30 22.61 120.87 18.07L120.87 18.07L125.12 15.54Q133.59 24.71 139.25 32.91L139.25 32.91L134.86 36.42Q132.47 32.77 131.77 31.68L131.77 31.68L129.80 31.89Q117.18 33.50 95.91 34.70L95.91 34.70L93.87 29.50Q99.67 29.39 101.57 29.29ZM99.91 39.66L133.49 39.66L133.49 68.38L128.39 68.38L128.39 64.51L105.01 64.51L105.01 68.38L99.91 68.38L99.91 39.66ZM128.39 44.05L105.01 44.05L105.01 60.05L128.39 60.05L128.39 44.05ZM95.70 16.84L91.79 21.16Q86.80 15.08 80.86 10.65L80.86 10.65L84.52 6.61Q90.14 10.65 95.70 16.84L95.70 16.84ZM92.00 33.47L88.10 37.79Q82.93 31.96 76.89 27.49L76.89 27.49L80.54 23.45Q87.01 28.05 92.00 33.47L92.00 33.47ZM81.42 67.82L77.17 63.21Q85.04 53.93 91.23 41.63L91.23 41.63L95.03 45.21Q88.63 58.46 81.42 67.82L81.42 67.82Z');
      uji.getD({
        anchor: 'middle'
      }, 'M38.46-32.41L38.46-22.46L65.00-22.46L65.00-8.02L59.59-8.02L59.59-17.93L12.27-17.93L12.27-8.02L6.93-8.02L6.93-22.46L32.98-22.46L32.98-32.41L38.46-32.41ZM56.46-5.27L39.13-5.27L39.13 5.52L67.68 5.52L67.68 10.20L39.13 10.20L39.13 27.35Q39.13 30.41 37.62 31.57L37.62 31.57Q36.35 32.52 32.98 32.52L32.98 32.52Q28.44 32.52 22.82 31.89L22.82 31.89L22.04 26.54Q28.41 27.42 31.68 27.42L31.68 27.42Q33.79 27.42 33.79 25.63L33.79 25.63L33.79 10.20L4.39 10.20L4.39 5.52L33.79 5.52L33.79-5.27L15.47-5.27L15.47-9.95L56.46-9.95L56.46-5.27ZM101.57-6.71L101.57-6.71L102.23-7.77Q108.14-18.53 112.18-31.04L112.18-31.04L117.77-29.39Q112.96-16.84 107.47-7.42L107.47-7.42L107.19-6.93L107.93-6.93Q114.79-7.21 122.45-7.95L122.45-7.95L128.88-8.44Q125.30-13.39 120.87-17.93L120.87-17.93L125.12-20.46Q133.59-11.29 139.25-3.09L139.25-3.09L134.86 0.42Q132.47-3.23 131.77-4.32L131.77-4.32L129.80-4.11Q117.18-2.50 95.91-1.30L95.91-1.30L93.87-6.50Q99.67-6.61 101.57-6.71ZM99.91 3.66L133.49 3.66L133.49 32.38L128.39 32.38L128.39 28.51L105.01 28.51L105.01 32.38L99.91 32.38L99.91 3.66ZM128.39 8.05L105.01 8.05L105.01 24.05L128.39 24.05L128.39 8.05ZM95.70-19.16L91.79-14.84Q86.80-20.92 80.86-25.35L80.86-25.35L84.52-29.39Q90.14-25.35 95.70-19.16L95.70-19.16ZM92.00-2.53L88.10 1.79Q82.93-4.04 76.89-8.51L76.89-8.51L80.54-12.55Q87.01-7.95 92.00-2.53L92.00-2.53ZM81.42 31.82L77.17 27.21Q85.04 17.93 91.23 5.63L91.23 5.63L95.03 9.21Q88.63 22.46 81.42 31.82L81.42 31.82Z');
      uji.getD({
        anchor: 'bottom'
      }, 'M38.46-68.41L38.46-58.46L65.00-58.46L65.00-44.02L59.59-44.02L59.59-53.93L12.27-53.93L12.27-44.02L6.93-44.02L6.93-58.46L32.98-58.46L32.98-68.41L38.46-68.41ZM56.46-41.27L39.13-41.27L39.13-30.48L67.68-30.48L67.68-25.80L39.13-25.80L39.13-8.65Q39.13-5.59 37.62-4.43L37.62-4.43Q36.35-3.48 32.98-3.48L32.98-3.48Q28.44-3.48 22.82-4.11L22.82-4.11L22.04-9.46Q28.41-8.58 31.68-8.58L31.68-8.58Q33.79-8.58 33.79-10.37L33.79-10.37L33.79-25.80L4.39-25.80L4.39-30.48L33.79-30.48L33.79-41.27L15.47-41.27L15.47-45.95L56.46-45.95L56.46-41.27ZM101.57-42.71L101.57-42.71L102.23-43.77Q108.14-54.53 112.18-67.04L112.18-67.04L117.77-65.39Q112.96-52.84 107.47-43.42L107.47-43.42L107.19-42.93L107.93-42.93Q114.79-43.21 122.45-43.95L122.45-43.95L128.88-44.44Q125.30-49.39 120.87-53.93L120.87-53.93L125.12-56.46Q133.59-47.29 139.25-39.09L139.25-39.09L134.86-35.58Q132.47-39.23 131.77-40.32L131.77-40.32L129.80-40.11Q117.18-38.50 95.91-37.30L95.91-37.30L93.87-42.50Q99.67-42.61 101.57-42.71ZM99.91-32.34L133.49-32.34L133.49-3.62L128.39-3.62L128.39-7.49L105.01-7.49L105.01-3.62L99.91-3.62L99.91-32.34ZM128.39-27.95L105.01-27.95L105.01-11.95L128.39-11.95L128.39-27.95ZM95.70-55.16L91.79-50.84Q86.80-56.92 80.86-61.35L80.86-61.35L84.52-65.39Q90.14-61.35 95.70-55.16L95.70-55.16ZM92.00-38.53L88.10-34.21Q82.93-40.04 76.89-44.51L76.89-44.51L80.54-48.55Q87.01-43.95 92.00-38.53L92.00-38.53ZM81.42-4.18L77.17-8.79Q85.04-18.07 91.23-30.38L91.23-30.38L95.03-26.79Q88.63-13.54 81.42-4.18L81.42-4.18Z');
      uji.getD({
        anchor: 'left top'
      }, 'M38.46 3.59L38.46 13.54L65.00 13.54L65.00 27.98L59.59 27.98L59.59 18.07L12.27 18.07L12.27 27.98L6.93 27.98L6.93 13.54L32.98 13.54L32.98 3.59L38.46 3.59ZM56.46 30.73L39.13 30.73L39.13 41.52L67.68 41.52L67.68 46.20L39.13 46.20L39.13 63.35Q39.13 66.41 37.62 67.57L37.62 67.57Q36.35 68.52 32.98 68.52L32.98 68.52Q28.44 68.52 22.82 67.89L22.82 67.89L22.04 62.54Q28.41 63.42 31.68 63.42L31.68 63.42Q33.79 63.42 33.79 61.63L33.79 61.63L33.79 46.20L4.39 46.20L4.39 41.52L33.79 41.52L33.79 30.73L15.47 30.73L15.47 26.05L56.46 26.05L56.46 30.73ZM101.57 29.29L101.57 29.29L102.23 28.23Q108.14 17.47 112.18 4.96L112.18 4.96L117.77 6.61Q112.96 19.16 107.47 28.58L107.47 28.58L107.19 29.07L107.93 29.07Q114.79 28.79 122.45 28.05L122.45 28.05L128.88 27.56Q125.30 22.61 120.87 18.07L120.87 18.07L125.12 15.54Q133.59 24.71 139.25 32.91L139.25 32.91L134.86 36.42Q132.47 32.77 131.77 31.68L131.77 31.68L129.80 31.89Q117.18 33.50 95.91 34.70L95.91 34.70L93.87 29.50Q99.67 29.39 101.57 29.29ZM99.91 39.66L133.49 39.66L133.49 68.38L128.39 68.38L128.39 64.51L105.01 64.51L105.01 68.38L99.91 68.38L99.91 39.66ZM128.39 44.05L105.01 44.05L105.01 60.05L128.39 60.05L128.39 44.05ZM95.70 16.84L91.79 21.16Q86.80 15.08 80.86 10.65L80.86 10.65L84.52 6.61Q90.14 10.65 95.70 16.84L95.70 16.84ZM92.00 33.47L88.10 37.79Q82.93 31.96 76.89 27.49L76.89 27.49L80.54 23.45Q87.01 28.05 92.00 33.47L92.00 33.47ZM81.42 67.82L77.17 63.21Q85.04 53.93 91.23 41.63L91.23 41.63L95.03 45.21Q88.63 58.46 81.42 67.82L81.42 67.82Z');
      uji.getD({
        anchor: 'left middle'
      }, 'M38.46-32.41L38.46-22.46L65.00-22.46L65.00-8.02L59.59-8.02L59.59-17.93L12.27-17.93L12.27-8.02L6.93-8.02L6.93-22.46L32.98-22.46L32.98-32.41L38.46-32.41ZM56.46-5.27L39.13-5.27L39.13 5.52L67.68 5.52L67.68 10.20L39.13 10.20L39.13 27.35Q39.13 30.41 37.62 31.57L37.62 31.57Q36.35 32.52 32.98 32.52L32.98 32.52Q28.44 32.52 22.82 31.89L22.82 31.89L22.04 26.54Q28.41 27.42 31.68 27.42L31.68 27.42Q33.79 27.42 33.79 25.63L33.79 25.63L33.79 10.20L4.39 10.20L4.39 5.52L33.79 5.52L33.79-5.27L15.47-5.27L15.47-9.95L56.46-9.95L56.46-5.27ZM101.57-6.71L101.57-6.71L102.23-7.77Q108.14-18.53 112.18-31.04L112.18-31.04L117.77-29.39Q112.96-16.84 107.47-7.42L107.47-7.42L107.19-6.93L107.93-6.93Q114.79-7.21 122.45-7.95L122.45-7.95L128.88-8.44Q125.30-13.39 120.87-17.93L120.87-17.93L125.12-20.46Q133.59-11.29 139.25-3.09L139.25-3.09L134.86 0.42Q132.47-3.23 131.77-4.32L131.77-4.32L129.80-4.11Q117.18-2.50 95.91-1.30L95.91-1.30L93.87-6.50Q99.67-6.61 101.57-6.71ZM99.91 3.66L133.49 3.66L133.49 32.38L128.39 32.38L128.39 28.51L105.01 28.51L105.01 32.38L99.91 32.38L99.91 3.66ZM128.39 8.05L105.01 8.05L105.01 24.05L128.39 24.05L128.39 8.05ZM95.70-19.16L91.79-14.84Q86.80-20.92 80.86-25.35L80.86-25.35L84.52-29.39Q90.14-25.35 95.70-19.16L95.70-19.16ZM92.00-2.53L88.10 1.79Q82.93-4.04 76.89-8.51L76.89-8.51L80.54-12.55Q87.01-7.95 92.00-2.53L92.00-2.53ZM81.42 31.82L77.17 27.21Q85.04 17.93 91.23 5.63L91.23 5.63L95.03 9.21Q88.63 22.46 81.42 31.82L81.42 31.82Z');
      uji.getD({
        anchor: 'left bottom'
      }, 'M38.46-68.41L38.46-58.46L65.00-58.46L65.00-44.02L59.59-44.02L59.59-53.93L12.27-53.93L12.27-44.02L6.93-44.02L6.93-58.46L32.98-58.46L32.98-68.41L38.46-68.41ZM56.46-41.27L39.13-41.27L39.13-30.48L67.68-30.48L67.68-25.80L39.13-25.80L39.13-8.65Q39.13-5.59 37.62-4.43L37.62-4.43Q36.35-3.48 32.98-3.48L32.98-3.48Q28.44-3.48 22.82-4.11L22.82-4.11L22.04-9.46Q28.41-8.58 31.68-8.58L31.68-8.58Q33.79-8.58 33.79-10.37L33.79-10.37L33.79-25.80L4.39-25.80L4.39-30.48L33.79-30.48L33.79-41.27L15.47-41.27L15.47-45.95L56.46-45.95L56.46-41.27ZM101.57-42.71L101.57-42.71L102.23-43.77Q108.14-54.53 112.18-67.04L112.18-67.04L117.77-65.39Q112.96-52.84 107.47-43.42L107.47-43.42L107.19-42.93L107.93-42.93Q114.79-43.21 122.45-43.95L122.45-43.95L128.88-44.44Q125.30-49.39 120.87-53.93L120.87-53.93L125.12-56.46Q133.59-47.29 139.25-39.09L139.25-39.09L134.86-35.58Q132.47-39.23 131.77-40.32L131.77-40.32L129.80-40.11Q117.18-38.50 95.91-37.30L95.91-37.30L93.87-42.50Q99.67-42.61 101.57-42.71ZM99.91-32.34L133.49-32.34L133.49-3.62L128.39-3.62L128.39-7.49L105.01-7.49L105.01-3.62L99.91-3.62L99.91-32.34ZM128.39-27.95L105.01-27.95L105.01-11.95L128.39-11.95L128.39-27.95ZM95.70-55.16L91.79-50.84Q86.80-56.92 80.86-61.35L80.86-61.35L84.52-65.39Q90.14-61.35 95.70-55.16L95.70-55.16ZM92.00-38.53L88.10-34.21Q82.93-40.04 76.89-44.51L76.89-44.51L80.54-48.55Q87.01-43.95 92.00-38.53L92.00-38.53ZM81.42-4.18L77.17-8.79Q85.04-18.07 91.23-30.38L91.23-30.38L95.03-26.79Q88.63-13.54 81.42-4.18L81.42-4.18Z');
      uji.getD({
        anchor: 'center top'
      }, 'M-33.54 3.59L-33.54 13.54L-7.00 13.54L-7.00 27.98L-12.41 27.98L-12.41 18.07L-59.73 18.07L-59.73 27.98L-65.07 27.98L-65.07 13.54L-39.02 13.54L-39.02 3.59L-33.54 3.59ZM-15.54 30.73L-32.87 30.73L-32.87 41.52L-4.32 41.52L-4.32 46.20L-32.87 46.20L-32.87 63.35Q-32.87 66.41-34.38 67.57L-34.38 67.57Q-35.65 68.52-39.02 68.52L-39.02 68.52Q-43.56 68.52-49.18 67.89L-49.18 67.89L-49.96 62.54Q-43.59 63.42-40.32 63.42L-40.32 63.42Q-38.21 63.42-38.21 61.63L-38.21 61.63L-38.21 46.20L-67.61 46.20L-67.61 41.52L-38.21 41.52L-38.21 30.73L-56.53 30.73L-56.53 26.05L-15.54 26.05L-15.54 30.73ZM29.57 29.29L29.57 29.29L30.23 28.23Q36.14 17.47 40.18 4.96L40.18 4.96L45.77 6.61Q40.96 19.16 35.47 28.58L35.47 28.58L35.19 29.07L35.93 29.07Q42.79 28.79 50.45 28.05L50.45 28.05L56.88 27.56Q53.30 22.61 48.87 18.07L48.87 18.07L53.12 15.54Q61.59 24.71 67.25 32.91L67.25 32.91L62.86 36.42Q60.47 32.77 59.77 31.68L59.77 31.68L57.80 31.89Q45.18 33.50 23.91 34.70L23.91 34.70L21.87 29.50Q27.67 29.39 29.57 29.29ZM27.91 39.66L61.49 39.66L61.49 68.38L56.39 68.38L56.39 64.51L33.01 64.51L33.01 68.38L27.91 68.38L27.91 39.66ZM56.39 44.05L33.01 44.05L33.01 60.05L56.39 60.05L56.39 44.05ZM23.70 16.84L19.79 21.16Q14.80 15.08 8.86 10.65L8.86 10.65L12.52 6.61Q18.14 10.65 23.70 16.84L23.70 16.84ZM20.00 33.47L16.10 37.79Q10.93 31.96 4.89 27.49L4.89 27.49L8.54 23.45Q15.01 28.05 20.00 33.47L20.00 33.47ZM9.42 67.82L5.17 63.21Q13.04 53.93 19.23 41.63L19.23 41.63L23.03 45.21Q16.63 58.46 9.42 67.82L9.42 67.82Z');
      uji.getD({
        anchor: 'center middle'
      }, 'M-33.54-32.41L-33.54-22.46L-7.00-22.46L-7.00-8.02L-12.41-8.02L-12.41-17.93L-59.73-17.93L-59.73-8.02L-65.07-8.02L-65.07-22.46L-39.02-22.46L-39.02-32.41L-33.54-32.41ZM-15.54-5.27L-32.87-5.27L-32.87 5.52L-4.32 5.52L-4.32 10.20L-32.87 10.20L-32.87 27.35Q-32.87 30.41-34.38 31.57L-34.38 31.57Q-35.65 32.52-39.02 32.52L-39.02 32.52Q-43.56 32.52-49.18 31.89L-49.18 31.89L-49.96 26.54Q-43.59 27.42-40.32 27.42L-40.32 27.42Q-38.21 27.42-38.21 25.63L-38.21 25.63L-38.21 10.20L-67.61 10.20L-67.61 5.52L-38.21 5.52L-38.21-5.27L-56.53-5.27L-56.53-9.95L-15.54-9.95L-15.54-5.27ZM29.57-6.71L29.57-6.71L30.23-7.77Q36.14-18.53 40.18-31.04L40.18-31.04L45.77-29.39Q40.96-16.84 35.47-7.42L35.47-7.42L35.19-6.93L35.93-6.93Q42.79-7.21 50.45-7.95L50.45-7.95L56.88-8.44Q53.30-13.39 48.87-17.93L48.87-17.93L53.12-20.46Q61.59-11.29 67.25-3.09L67.25-3.09L62.86 0.42Q60.47-3.23 59.77-4.32L59.77-4.32L57.80-4.11Q45.18-2.50 23.91-1.30L23.91-1.30L21.87-6.50Q27.67-6.61 29.57-6.71ZM27.91 3.66L61.49 3.66L61.49 32.38L56.39 32.38L56.39 28.51L33.01 28.51L33.01 32.38L27.91 32.38L27.91 3.66ZM56.39 8.05L33.01 8.05L33.01 24.05L56.39 24.05L56.39 8.05ZM23.70-19.16L19.79-14.84Q14.80-20.92 8.86-25.35L8.86-25.35L12.52-29.39Q18.14-25.35 23.70-19.16L23.70-19.16ZM20.00-2.53L16.10 1.79Q10.93-4.04 4.89-8.51L4.89-8.51L8.54-12.55Q15.01-7.95 20.00-2.53L20.00-2.53ZM9.42 31.82L5.17 27.21Q13.04 17.93 19.23 5.63L19.23 5.63L23.03 9.21Q16.63 22.46 9.42 31.82L9.42 31.82Z');
      uji.getD({
        anchor: 'center bottom'
      }, 'M-33.54-68.41L-33.54-58.46L-7.00-58.46L-7.00-44.02L-12.41-44.02L-12.41-53.93L-59.73-53.93L-59.73-44.02L-65.07-44.02L-65.07-58.46L-39.02-58.46L-39.02-68.41L-33.54-68.41ZM-15.54-41.27L-32.87-41.27L-32.87-30.48L-4.32-30.48L-4.32-25.80L-32.87-25.80L-32.87-8.65Q-32.87-5.59-34.38-4.43L-34.38-4.43Q-35.65-3.48-39.02-3.48L-39.02-3.48Q-43.56-3.48-49.18-4.11L-49.18-4.11L-49.96-9.46Q-43.59-8.58-40.32-8.58L-40.32-8.58Q-38.21-8.58-38.21-10.37L-38.21-10.37L-38.21-25.80L-67.61-25.80L-67.61-30.48L-38.21-30.48L-38.21-41.27L-56.53-41.27L-56.53-45.95L-15.54-45.95L-15.54-41.27ZM29.57-42.71L29.57-42.71L30.23-43.77Q36.14-54.53 40.18-67.04L40.18-67.04L45.77-65.39Q40.96-52.84 35.47-43.42L35.47-43.42L35.19-42.93L35.93-42.93Q42.79-43.21 50.45-43.95L50.45-43.95L56.88-44.44Q53.30-49.39 48.87-53.93L48.87-53.93L53.12-56.46Q61.59-47.29 67.25-39.09L67.25-39.09L62.86-35.58Q60.47-39.23 59.77-40.32L59.77-40.32L57.80-40.11Q45.18-38.50 23.91-37.30L23.91-37.30L21.87-42.50Q27.67-42.61 29.57-42.71ZM27.91-32.34L61.49-32.34L61.49-3.62L56.39-3.62L56.39-7.49L33.01-7.49L33.01-3.62L27.91-3.62L27.91-32.34ZM56.39-27.95L33.01-27.95L33.01-11.95L56.39-11.95L56.39-27.95ZM23.70-55.16L19.79-50.84Q14.80-56.92 8.86-61.35L8.86-61.35L12.52-65.39Q18.14-61.35 23.70-55.16L23.70-55.16ZM20.00-38.53L16.10-34.21Q10.93-40.04 4.89-44.51L4.89-44.51L8.54-48.55Q15.01-43.95 20.00-38.53L20.00-38.53ZM9.42-4.18L5.17-8.79Q13.04-18.07 19.23-30.38L19.23-30.38L23.03-26.79Q16.63-13.54 9.42-4.18L9.42-4.18Z');
      uji.getD({
        anchor: 'right top'
      }, 'M-105.54 3.59L-105.54 13.54L-79.00 13.54L-79.00 27.98L-84.41 27.98L-84.41 18.07L-131.73 18.07L-131.73 27.98L-137.07 27.98L-137.07 13.54L-111.02 13.54L-111.02 3.59L-105.54 3.59ZM-87.54 30.73L-104.87 30.73L-104.87 41.52L-76.32 41.52L-76.32 46.20L-104.87 46.20L-104.87 63.35Q-104.87 66.41-106.38 67.57L-106.38 67.57Q-107.65 68.52-111.02 68.52L-111.02 68.52Q-115.56 68.52-121.18 67.89L-121.18 67.89L-121.96 62.54Q-115.59 63.42-112.32 63.42L-112.32 63.42Q-110.21 63.42-110.21 61.63L-110.21 61.63L-110.21 46.20L-139.61 46.20L-139.61 41.52L-110.21 41.52L-110.21 30.73L-128.53 30.73L-128.53 26.05L-87.54 26.05L-87.54 30.73ZM-42.43 29.29L-42.43 29.29L-41.77 28.23Q-35.86 17.47-31.82 4.96L-31.82 4.96L-26.23 6.61Q-31.04 19.16-36.53 28.58L-36.53 28.58L-36.81 29.07L-36.07 29.07Q-29.21 28.79-21.55 28.05L-21.55 28.05L-15.12 27.56Q-18.70 22.61-23.13 18.07L-23.13 18.07L-18.88 15.54Q-10.41 24.71-4.75 32.91L-4.75 32.91L-9.14 36.42Q-11.53 32.77-12.23 31.68L-12.23 31.68L-14.20 31.89Q-26.82 33.50-48.09 34.70L-48.09 34.70L-50.13 29.50Q-44.33 29.39-42.43 29.29ZM-44.09 39.66L-10.51 39.66L-10.51 68.38L-15.61 68.38L-15.61 64.51L-38.99 64.51L-38.99 68.38L-44.09 68.38L-44.09 39.66ZM-15.61 44.05L-38.99 44.05L-38.99 60.05L-15.61 60.05L-15.61 44.05ZM-48.30 16.84L-52.21 21.16Q-57.20 15.08-63.14 10.65L-63.14 10.65L-59.48 6.61Q-53.86 10.65-48.30 16.84L-48.30 16.84ZM-52.00 33.47L-55.90 37.79Q-61.07 31.96-67.11 27.49L-67.11 27.49L-63.46 23.45Q-56.99 28.05-52.00 33.47L-52.00 33.47ZM-62.58 67.82L-66.83 63.21Q-58.96 53.93-52.77 41.63L-52.77 41.63L-48.97 45.21Q-55.37 58.46-62.58 67.82L-62.58 67.82Z');
      uji.getD({
        anchor: 'right middle'
      }, 'M-105.54-32.41L-105.54-22.46L-79.00-22.46L-79.00-8.02L-84.41-8.02L-84.41-17.93L-131.73-17.93L-131.73-8.02L-137.07-8.02L-137.07-22.46L-111.02-22.46L-111.02-32.41L-105.54-32.41ZM-87.54-5.27L-104.87-5.27L-104.87 5.52L-76.32 5.52L-76.32 10.20L-104.87 10.20L-104.87 27.35Q-104.87 30.41-106.38 31.57L-106.38 31.57Q-107.65 32.52-111.02 32.52L-111.02 32.52Q-115.56 32.52-121.18 31.89L-121.18 31.89L-121.96 26.54Q-115.59 27.42-112.32 27.42L-112.32 27.42Q-110.21 27.42-110.21 25.63L-110.21 25.63L-110.21 10.20L-139.61 10.20L-139.61 5.52L-110.21 5.52L-110.21-5.27L-128.53-5.27L-128.53-9.95L-87.54-9.95L-87.54-5.27ZM-42.43-6.71L-42.43-6.71L-41.77-7.77Q-35.86-18.53-31.82-31.04L-31.82-31.04L-26.23-29.39Q-31.04-16.84-36.53-7.42L-36.53-7.42L-36.81-6.93L-36.07-6.93Q-29.21-7.21-21.55-7.95L-21.55-7.95L-15.12-8.44Q-18.70-13.39-23.13-17.93L-23.13-17.93L-18.88-20.46Q-10.41-11.29-4.75-3.09L-4.75-3.09L-9.14 0.42Q-11.53-3.23-12.23-4.32L-12.23-4.32L-14.20-4.11Q-26.82-2.50-48.09-1.30L-48.09-1.30L-50.13-6.50Q-44.33-6.61-42.43-6.71ZM-44.09 3.66L-10.51 3.66L-10.51 32.38L-15.61 32.38L-15.61 28.51L-38.99 28.51L-38.99 32.38L-44.09 32.38L-44.09 3.66ZM-15.61 8.05L-38.99 8.05L-38.99 24.05L-15.61 24.05L-15.61 8.05ZM-48.30-19.16L-52.21-14.84Q-57.20-20.92-63.14-25.35L-63.14-25.35L-59.48-29.39Q-53.86-25.35-48.30-19.16L-48.30-19.16ZM-52.00-2.53L-55.90 1.79Q-61.07-4.04-67.11-8.51L-67.11-8.51L-63.46-12.55Q-56.99-7.95-52.00-2.53L-52.00-2.53ZM-62.58 31.82L-66.83 27.21Q-58.96 17.93-52.77 5.63L-52.77 5.63L-48.97 9.21Q-55.37 22.46-62.58 31.82L-62.58 31.82Z');
      uji.getD({
        anchor: 'right bottom'
      }, 'M-105.54-68.41L-105.54-58.46L-79.00-58.46L-79.00-44.02L-84.41-44.02L-84.41-53.93L-131.73-53.93L-131.73-44.02L-137.07-44.02L-137.07-58.46L-111.02-58.46L-111.02-68.41L-105.54-68.41ZM-87.54-41.27L-104.87-41.27L-104.87-30.48L-76.32-30.48L-76.32-25.80L-104.87-25.80L-104.87-8.65Q-104.87-5.59-106.38-4.43L-106.38-4.43Q-107.65-3.48-111.02-3.48L-111.02-3.48Q-115.56-3.48-121.18-4.11L-121.18-4.11L-121.96-9.46Q-115.59-8.58-112.32-8.58L-112.32-8.58Q-110.21-8.58-110.21-10.37L-110.21-10.37L-110.21-25.80L-139.61-25.80L-139.61-30.48L-110.21-30.48L-110.21-41.27L-128.53-41.27L-128.53-45.95L-87.54-45.95L-87.54-41.27ZM-42.43-42.71L-42.43-42.71L-41.77-43.77Q-35.86-54.53-31.82-67.04L-31.82-67.04L-26.23-65.39Q-31.04-52.84-36.53-43.42L-36.53-43.42L-36.81-42.93L-36.07-42.93Q-29.21-43.21-21.55-43.95L-21.55-43.95L-15.12-44.44Q-18.70-49.39-23.13-53.93L-23.13-53.93L-18.88-56.46Q-10.41-47.29-4.75-39.09L-4.75-39.09L-9.14-35.58Q-11.53-39.23-12.23-40.32L-12.23-40.32L-14.20-40.11Q-26.82-38.50-48.09-37.30L-48.09-37.30L-50.13-42.50Q-44.33-42.61-42.43-42.71ZM-44.09-32.34L-10.51-32.34L-10.51-3.62L-15.61-3.62L-15.61-7.49L-38.99-7.49L-38.99-3.62L-44.09-3.62L-44.09-32.34ZM-15.61-27.95L-38.99-27.95L-38.99-11.95L-15.61-11.95L-15.61-27.95ZM-48.30-55.16L-52.21-50.84Q-57.20-56.92-63.14-61.35L-63.14-61.35L-59.48-65.39Q-53.86-61.35-48.30-55.16L-48.30-55.16ZM-52.00-38.53L-55.90-34.21Q-61.07-40.04-67.11-44.51L-67.11-44.51L-63.46-48.55Q-56.99-43.95-52.00-38.53L-52.00-38.53ZM-62.58-4.18L-66.83-8.79Q-58.96-18.07-52.77-30.38L-52.77-30.38L-48.97-26.79Q-55.37-13.54-62.58-4.18L-62.58-4.18Z');
    });
  });
});